/*************************************************************************/
/*************************************************************************/
/*                                                                       */
/*                 THE HEI-2020 SCORING MACRO                            */                                               
/*                  (hei2020.score.macro.sas)                            */
/*************************************************************************/
/*                     VERSION 1.0        08/17/2023                     */   
/*                                                                       */
/*                                                                       */
/* This HEI-2020 macro is to be used to calculate densities and          */
/*  HEI-2020 component and total scores for subjects age >= 2 years old. */
/*								         */	
/* The macro requires an input dataset with variables for each of        */
/* the HEI-2020 components, noted below.                                 */
/*								         */	
/* The resulting dataset, which is named by the user, contains the       */
/* same variables as the supplied dataset, and creates 27 new            */
/* variables. These include:					         */
/*                                                                       */				    
/*   The densities (per 1000 kcal) or percent (of total calories)        */
/*    for each of the 13 HEI-2020 components.			         */
/*                                                                       */
/*   Scores for the 13 components of the HEI-2020.                       */
/*                                                                       */
/*   The total HEI-2020 score, which is the sum of the                   */
/*    scores for the 13 components.                                      */
/*                                                                       */                                                                  
/* The syntax for calling the macro is:                                  */
/*                                                                       */
/* %HEI2020   	 						         */
/* (indat=,kcal=,f_total=,fwholefrt=,vtotalleg=,vdrkgrleg=,g_whole=      */
/*    d_total=,pfallprotleg=,pfseaplantleg=,monopoly=,g_refined=,sodium=,*/
/*    add_sugars=,satfat=,outdat=);                                      */
/*                                                                       */
/*  where                                                                */
/*                                                                       */
/*   "indat"        * Specifies the dataset to be used.                  */
/*                                                                       */
/*   "kcal"         * Specifies calorie amount.                          */
/*                                                                       */
/*   "f_total"      * Specifies the intake of total fruit in cup eq.     */
/*                                                                       */
/*   "fwholefrt"    * Specifies the intake of whole fruit in cup eq.     */
/*                                                                       */
/*   "vtotalleg"    * Specifies the intake of total veg plus             */
/*                      legumes in cup eq.                               */
/*                                                                       */
/*   "vdrkgrleg"    * Specifies the intake of dark green veg             */
/*                      plus legumes in cup eq.                          */
/*                                                                       */
/*   "g_whole"      * Specifies the intake of whole grain in oz. eq.     */
/*                                                                       */
/*   "d_total"      * Specifies the intake of total dairy in cup eq.     */
/*                                                                       */
/*   "pfallprotleg" * Specifies the intake of total protein foods        */
/*                      (includes legumes) in oz. eq.                    */
/*                                                                       */
/*   "pfseaplantleg"  * Specifies the intake of seafood, fish and plant  */
/*                      protein foods (includes legumes) in oz. eq.      */
/*                                                                       */
/*   "monopoly"       * Specifies the grams of mono fat plus poly fat.   */
/*                                                                       */
/*   "g_refined"      * Specifies the intake of refined                  */
/*                       grain in oz. eq.                                */
/*                                                                       */
/*   "sodium"         * Specifies the mg of sodium.                      */
/*                                                                       */                                                                 
/*   "add_sugars"     * Specifies the intake of added sugars in tsp. eq. */
/*                                                                       */
/*   "satfat"         * Specifies the grams of saturated fat.            */
/*                                                                       */
/*   "outdat"         * Specifies the name of the resulting dataset.     */
/*								         */
/*                                                                       */                                                                 
/* Caution:  variable names "FARMIN", "FARMAX", "SODMIN",                */
/*   "SODMAX", "RGMIN", "RGMAX", "SFATMIN", "SFATMAX", "ADDSUGMIN",      */
/*   "ADDSUGMAX" are reserved for this macro.                            */
/*                                                                       */
/*                                                                       */
/*************************************************************************/
;

     


%macro HEI2020 (indat=,kcal=,f_total=,fwholefrt=,vtotalleg=,vdrkgrleg=,g_whole=,     
    d_total=,pfallprotleg=,pfseaplantleg=,monopoly=,g_refined=,sodium=,add_sugars=,satfat=,outdat=);

data &outdat (drop=FARMIN FARMAX SODMAX SODMIN RGMIN RGMAX SFATMIN SFATMAX ADDSUGMIN ADDSUGMAX);
  set &indat;

/*********************************************************************************************/
/*  This section of the macro is for scoring Adequacy components; higher 		     */
/*  reported density-based intakes (variables containing "DEN") of these 		     */
/*  components results in higher scores (up to the maximum points for that component).	     */
/*********************************************************************************************/

  IF &kcal > 0 then FRTDEN=&f_total/(&kcal/1000);
  HEI2020_TOTALFRUIT=5*(FRTDEN/0.8);
  IF HEI2020_TOTALFRUIT > 5 THEN HEI2020_TOTALFRUIT=5;
  IF FRTDEN=0 THEN HEI2020_TOTALFRUIT=0;	

  IF &kcal > 0 then WHFRDEN=&fwholefrt/(&kcal/1000);
  HEI2020_WHOLEFRUIT=5*(WHFRDEN/0.4); 
  IF HEI2020_WHOLEFRUIT > 5 THEN HEI2020_WHOLEFRUIT=5;
  IF WHFRDEN=0 THEN HEI2020_WHOLEFRUIT=0;	

  IF &kcal > 0 then VEGDEN=&vtotalleg/(&kcal/1000);
  HEI2020_TOTALVEG=5*(VEGDEN/1.1);
  IF HEI2020_TOTALVEG > 5 THEN HEI2020_TOTALVEG=5;
  IF VEGDEN=0 THEN HEI2020_TOTALVEG=0;

  IF &kcal > 0 then GRBNDEN=&vdrkgrleg/(&kcal/1000);
  HEI2020_GREEN_AND_BEAN=5*(GRBNDEN/0.2);
  IF HEI2020_GREEN_AND_BEAN > 5 THEN HEI2020_GREEN_AND_BEAN=5;
  IF GRBNDEN=0 THEN HEI2020_GREEN_AND_BEAN=0;

  IF &kcal > 0 then WGRNDEN=&g_whole/(&kcal/1000);
  HEI2020_WHOLEGRAIN=10*(WGRNDEN/1.5);
  IF HEI2020_WHOLEGRAIN > 10 THEN HEI2020_WHOLEGRAIN=10;
  IF WGRNDEN=0 THEN HEI2020_WHOLEGRAIN=0;

  IF &kcal > 0 then DAIRYDEN=&d_total/(&kcal/1000);
  HEI2020_TOTALDAIRY=10*(DAIRYDEN/1.3);
  IF HEI2020_TOTALDAIRY > 10 THEN HEI2020_TOTALDAIRY=10;
  IF DAIRYDEN=0 THEN HEI2020_TOTALDAIRY=0;

  IF &kcal > 0 then PROTDEN=&pfallprotleg/(&kcal/1000);
  HEI2020_TOTPROT=5*(PROTDEN/2.5);
  IF HEI2020_TOTPROT > 5 THEN HEI2020_TOTPROT=5;
  IF PROTDEN=0 THEN HEI2020_TOTPROT=0;

  IF &kcal > 0 then SEAPLDEN=&pfseaplantleg/(&kcal/1000);
  HEI2020_SEAPLANT_PROT=5*(SEAPLDEN/0.8);
  IF HEI2020_SEAPLANT_PROT > 5 THEN HEI2020_SEAPLANT_PROT=5;
  IF SEAPLDEN=0 THEN HEI2020_SEAPLANT_PROT=0;


/**************************************************************************************************/
/*  The Fatty Acid component is calculated as a ratio; variables listed as "MAX" and "MIN"        */
/*  here relate to the ratio of unsaturated to saturated fat; thus the "MIN" value corresponds 	  */
/*  to the minimum point standard and the "MAX" value corresponds to the maximum point standard.  */
/**************************************************************************************************/


  IF &satfat > 0 THEN FARATIO=&monopoly/&satfat;
  FARMIN=1.2;
  FARMAX=2.5;
  if &satfat=0 and &monopoly=0 then HEI2020_FATTYACID=0;
    else if &satfat=0 and &monopoly > 0 then HEI2020_FATTYACID=10;
    else if FARATIO >= FARMAX THEN HEI2020_FATTYACID=10;
    else if FARATIO <= FARMIN THEN HEI2020_FATTYACID=0;
    else HEI2020_FATTYACID=10* ( (FARATIO-FARMIN) / (FARMAX-FARMIN) );


/************************************************************************************************/
/*  The following section of the macro is for scoring Moderation components; higher reported    */
/*  intake of these food components results in LOWER scores and lower reported intake results   */
/*  in HIGHER scores (up to the maximum points for that component). Variables below labeled     */
/*  "MIN" or "MAX" relate to intake amounts; thus "MIN" values correspond to the maximum point  */
/*  standard and "MAX" values correspond to the minimum point standard for that component.      */
/************************************************************************************************/


  IF &kcal > 0 then RGDEN=&g_refined/(&kcal/1000);
  RGMIN=1.8;
  RGMAX=4.3;
  IF RGDEN <= RGMIN THEN HEI2020_REFINEDGRAIN=10;
    ELSE IF RGDEN >= RGMAX THEN HEI2020_REFINEDGRAIN=0;
    ELSE HEI2020_REFINEDGRAIN=10 - ( 10* (RGDEN-RGMIN) / (RGMAX-RGMIN) ); 

  IF &kcal > 0 then SODDEN=&sodium/&kcal;
  SODMIN=1.1;
  SODMAX=2.0;
  IF SODDEN <= SODMIN THEN HEI2020_SODIUM=10;
    ELSE IF SODDEN >= SODMAX THEN HEI2020_SODIUM=0;
    ELSE HEI2020_SODIUM=10 - (10 * (SODDEN-SODMIN) / (SODMAX-SODMIN) );

  IF &kcal > 0 then ADDSUG_PERC=100*(&add_sugars*16/&kcal); 
  ADDSUGMIN=6.5;
  ADDSUGMAX=26;
  IF ADDSUG_PERC >= ADDSUGMAX THEN HEI2020_ADDSUG=0;
    ELSE IF ADDSUG_PERC <= ADDSUGMIN THEN HEI2020_ADDSUG=10;
    ELSE HEI2020_ADDSUG= 10 - ( 10* (ADDSUG_PERC-ADDSUGMIN) / (ADDSUGMAX-ADDSUGMIN) );
 
  IF &kcal > 0 then SFAT_PERC=100*(&satfat*9/&kcal); 
  SFATMIN=8;
  SFATMAX=16;
  IF SFAT_PERC >= SFATMAX THEN HEI2020_SFAT=0;
    ELSE IF SFAT_PERC <= SFATMIN THEN HEI2020_SFAT=10;
    ELSE HEI2020_SFAT= 10 - ( 10* (SFAT_PERC-SFATMIN) / (SFATMAX-SFATMIN) );

IF &kcal=0 THEN DO;
  HEI2020_TOTALFRUIT=0; HEI2020_WHOLEFRUIT=0; HEI2020_TOTALVEG=0; HEI2020_GREEN_AND_BEAN=0;  HEI2020_WHOLEGRAIN=0; HEI2020_TOTALDAIRY=0;
  HEI2020_TOTPROT=0;  HEI2020_SEAPLANT_PROT=0; HEI2020_FATTYACID=0; HEI2020_REFINEDGRAIN=0; HEI2020_SODIUM=0;  HEI2020_ADDSUG=0; HEI2020_SFAT=0;
  END;

/**Calculate HEI-2020 total score**/
/*total HEI-2020 score is the sum of 13 HEI component scores*/

HEI2020_TOTAL_SCORE = HEI2020_TOTALFRUIT + HEI2020_WHOLEFRUIT + HEI2020_TOTALVEG + HEI2020_GREEN_AND_BEAN + HEI2020_WHOLEGRAIN + HEI2020_TOTALDAIRY +
  HEI2020_TOTPROT + HEI2020_SEAPLANT_PROT + HEI2020_FATTYACID + HEI2020_REFINEDGRAIN + HEI2020_SODIUM  + HEI2020_ADDSUG + HEI2020_SFAT;


LABEL HEI2020_TOTAL_SCORE='TOTAL HEI-2020 SCORE'
      HEI2020_TOTALFRUIT='HEI-2020 COMPONENT TOTAL FRUIT'
      HEI2020_WHOLEFRUIT='HEI-2020 COMPONENT WHOLE FRUIT'
      HEI2020_TOTALVEG='HEI-2020 COMPONENT TOTAL VEGETABLES'
      HEI2020_GREEN_AND_BEAN='HEI-2020 COMPONENT GREENS AND BEANS'
      HEI2020_WHOLEGRAIN='HEI-2020 COMPONENT WHOLE GRAINS'
      HEI2020_TOTALDAIRY='HEI-2020 COMPONENT DAIRY'
      HEI2020_TOTPROT='HEI-2020 COMPONENT TOTAL PROTEIN FOODS'
      HEI2020_SEAPLANT_PROT='HEI-2020 COMPONENT SEAFOOD AND PLANT PROTEIN'
      HEI2020_FATTYACID='HEI-2020 COMPONENT FATTY ACID RATIO'
      HEI2020_REFINEDGRAIN='HEI-2020 COMPONENT REFINED GRAINS'
      HEI2020_SODIUM='HEI-2020 COMPONENT SODIUM'
      HEI2020_ADDSUG='HEI-2020 COMPONENT ADDED SUGAR'
      HEI2020_SFAT='HEI-2020 COMPONENT SAT FAT'

      FRTDEN='DENSITY OF TOTAL FRUIT PER 1000 KCAL'
      WHFRDEN='DENSITY OF WHOLE FRUIT PER 1000 KCAL'
      VEGDEN='DENSITY OF TOTAL VEGETABLES PER 1000 KCAL'
      GRBNDEN='DENSITY OF DARK GREEN VEG AND BEANS PER 1000 KCAL'
      WGRNDEN='DENSITY OF WHOLE GRAIN PER 1000 KCAL'
      DAIRYDEN='DENSITY OF DAIRY PER 1000 KCAL'
      PROTDEN='DENSITY OF TOTAL PROTEIN PER 1000 KCAL'
      SEAPLDEN='DENSITY OF SEAFOOD AND PLANT PROTEIN PER 1000 KCAL'
      FARATIO='FATTY ACID RATIO'
      RGDEN='DENSITY OF REFINED GRAINS PER 1000 KCAL'
      SODDEN='DENSITY OF SODIUM PER 1000 KCAL'
      ADDSUG_PERC='PERCENT OF CALORIES FROM ADDED SUGAR'
      SFAT_PERC='PERCENT OF CALORIES FROM SAT FAT'
      ;

run;

%mend HEI2020;


/*******************************************************************/
/*  END OF THE HEI2020 MACRO                                       */
/*******************************************************************/



