**Creates logger dataset to save notes, warnings, and errors;
%macro create_logger();

	data _log_record;
		set _NULL_;
		
		length section $128.;
		length message $32767.;
	run;
%mend create_logger;

**Adds notes, warnings, and errors to the logger dataset and writes them to the log;
%macro update_logger(error_code=0,
										 section=,
										 message=);
										 
	%local error_class;
										 
	**Update logger dataset;
	data _new_record;
	
		length section $128.;
		length message $32767.;
		
		error_code = &error_code.;
		section = "&section.";
		message = "&message.";
	run;
	
	data _log_record;
		set _log_record
				_new_record;
	run;
	
	**Write to log;
	%if &error_code. > 0 %then %do;
	
		%if &error_code. = 1 %then %do;
	
			%let error_class = NOTE;
		%end;
		%else %if &error_code. = 2 %then %do;
	
			%let error_class = WARNING;
		%end;
		%else %do;
	
			%let error_class = ERROR;
		%end;
	
		%let message = &error_class.: &message.;
	
		%put &message.;
	%end;									 
%mend update_logger;

**Checks variables for valid matches in a dataset;
%macro check_variables_data(variables=,
														variable_text=,
														dataset=,
														dataset_text=,
														section=);
														
	%local i variable var_exists;
														
	proc contents data=&dataset. out=_data_contents noprint; run;

	%do i = 1 %to %sysfunc(countw(&variables., %str( )));
	
		%let variable = %sysfunc(scan(&variables., &i., %str( )));
		
		%let var_exists = 0;
		data _NULL_;
			set _data_contents;
			
			if compare("&variable.", NAME, "i") = 0 then call symputx("var_exists", 1);
		run;
		
		%if &var_exists. = 1 %then %do;
		
			%update_logger(error_code=0, section=&section., message=&variable_text. &variable. was found in &dataset_text.);
		%end;
		%else %do;
		
			%update_logger(error_code=3, section=&section., message=&variable_text. &variable. was not found in &dataset_text.);
		%end;
	%end;									 
%mend check_variables_data;

**Checks variables for valid matches in a data column;
%macro check_variables_column(variables=,
															variable_text=,
															dataset=,
															column=,
															column_text=,
															section=);
															
	%local i variable var_exists;
															
	%do i = 1 %to %sysfunc(countw(&variables., %str( )));
	
		%let variable = %sysfunc(scan(&variables., &i., %str( )));
		
		%let var_exists = 0;
		data _NULL_;
			set &dataset. end=_last;
			
			if compare("&variable.", &column., "i") = 0 then call symputx("var_exists", 1);
		run;
		
		%if &var_exists. = 1 %then %do;
		
			%update_logger(error_code=0, section=&section., message=&variable_text. &variable. was found in &column_text.);
		%end;
		%else %do;
		
			%update_logger(error_code=3, section=&section., message=&variable_text. &variable. was not found in &column_text.);
		%end;
	%end;
%mend check_variables_column;

**Checks variables for valid matches in a macro list;
%macro check_variables_macro_list(variables=,
																	variable_text=,
																	list=,
																	list_text=,
																	section=);
																	
	%local i variable var_exists;
																	
	%do i = 1 %to %sysfunc(countw(&variables., %str( )));
	
		%let variable = %sysfunc(scan(&variables., &i., %str( )));
		
		%let var_exists = 0;
		data _NULL_;
		
			if findw("&list.", "&variable.", " ", "ei") then call symputx("var_exists", 1);
		run;
		
		%if &var_exists. = 1 %then %do;
		
			%update_logger(error_code=0, section=&section., message=&variable_text. &variable. was found in &list_text.);
		%end;
		%else %do;
		
			%update_logger(error_code=3, section=&section., message=&variable_text. &variable. was not found in &list_text.);
		%end;
	%end;
%mend check_variables_macro_list;

**Checks indicators and amounts for corresponding variables in backtransformation;
%macro check_backtran(variables=,
											backtran=,
											section=);
											
	%local i variable var_exists;

	%do i = 1 %to %sysfunc(countw(&variables., %str( )));
	
		%let variable = %sysfunc(scan(&variables., &i., %str( )));
		
		%let var_exists = 0;
		data _NULL_;
			set &backtran.;
			
			if find("&variable.", variable, "i") ^= 0 then call symputx("var_exists", 1);
		run;
		
		%if &var_exists. = 1 %then %do;
		
			%update_logger(error_code=0, section=&section., message=&variable. was found in backtransformation variables);
		%end;
		%else %do;
		
			%update_logger(error_code=3, section=&section., message=&variable. was not found in backtransformation variables);
		%end;
	%end;
%mend check_backtran;

**Checks number of MCMC iterations, burn-in, and thinning;
%macro check_iterations(num_mcmc_iterations=,
												num_burn=,
												num_thin=,
												num_post=);
					
	**Number of MCMC iterations;							
	%if &num_mcmc_iterations. = %str() %then %do;
	
		%update_logger(error_code=3, section=Iterations, message=Number of MCMC iterations not specified);
	%end;
	%else %do;
	
		%update_logger(error_code=0, section=Iterations, message=Number of MCMC iterations specified);
	%end;
	
	%if &num_mcmc_iterations. ^= %str() %then %do;
	
		%if &num_mcmc_iterations. < 1 %then %do;
		
			%update_logger(error_code=3, section=Iterations, message=Number of MCMC iterations is less than one);
		%end;
		%else %do;
		
			%update_logger(error_code=0, section=Iterations, message=Number of MCMC iterations is at least one);
		%end;
	%end;
	
	**Number of burn-in iterations;
	%if &num_burn. = %str() %then %do;
	
		%update_logger(error_code=3, section=Iterations, message=Number of burn-in iterations not specified);
	%end;
	%else %do;
	
		%update_logger(error_code=0, section=Iterations, message=Number of burn-in iterations specified);
	%end;
	
	%if &num_burn. ^= %str() %then %do;
	
		%if &num_burn. < 0 %then %do;
		
			%update_logger(error_code=3, section=Iterations, message=Number of burn-in iterations is negative);
		%end;
		%else %do;
		
			%update_logger(error_code=0, section=Iterations, message=Number of burn-in iterations is not negative);
		%end;
	%end;
	
	**Compare total iterations and burn-in;
	%if &num_mcmc_iterations. ^= %str() and &num_burn. ^= %str() %then %do;
	
		%if &num_mcmc_iterations. <= &num_burn. %then %do;
		
			%update_logger(error_code=3, section=Iterations, message=Number of burn-in iterations is not less than total number of MCMC iterations);
		%end;
		%else %do;
		
			%update_logger(error_code=0, section=Iterations, message=Number of burn-in iterations is less than total number of MCMC iterations);
		%end;
	%end;
	%else %do;
	
		%update_logger(error_code=1, section=Iterations, message=Number of MCMC iterations not compared to number of burn-in iterations because one or both were not specified);
	%end;
	
	**Thinning number;
	%if &num_thin. = %str() %then %do;
	
		%update_logger(error_code=3, section=Iterations, message=Thinning number not specified);
	%end;
	%else %do;
	
		%update_logger(error_code=0, section=Iterations, message=Thinning number specified);
	%end;
	
	%if &num_thin. ^= %str() %then %do;
	
		%if &num_thin. < 1 %then %do;
		
			%update_logger(error_code=3, section=Iterations, message=Thinning number is less than one);
		%end;
		%else %do;
		
			%update_logger(error_code=0, section=Iterations, message=Thinning number is at least one);
		%end;
	%end;
	
	**Post-MCMC iterations;
	%if &num_post. = %str() %then %do;
	
		%update_logger(error_code=3, section=Iterations, message=Number of post-MCMC iterations not specified);
	%end;
	%else %do;
	
		%update_logger(error_code=0, section=Iterations, message=Number of post-MCMC iterations specified);
	%end;
	
	%if &num_post. ^= %str() %then %do;
	
		%if &num_post. < 0 %then %do;
		
			%update_logger(error_code=3, section=Iterations, message=Number of post-MCMC iterations is negative);
		%end;
		%else %do;
		
			%update_logger(error_code=0, section=Iterations, message=Number of post-MCMC iterations is not negative);
		%end;
	%end;
%mend check_iterations;

**Check MCMC input library;
%macro check_data(pre_mcmc_data=,
									id=,
									repeat_obs=,
									weight=);

	**Input dataset;
	%if %sysfunc(exist(&pre_mcmc_data._mcmc_in)) = 0 %then %do;
	
		%update_logger(error_code=3, section=Data, message=&pre_mcmc_data._mcmc_in not found.);
	%end;
	%else %do;
	
		%update_logger(error_code=0, section=Data, message=&pre_mcmc_data._mcmc_in found.);
	%end;
	
	%if &id. = %str() %then %do;
	
		%update_logger(error_code=3, section=ID, message=Subject ID variable not specified);
	%end;
	%else %do;
	
		%update_logger(error_code=0, section=ID, message=Subject ID variable specified);
	%end;
	
	%if %sysfunc(exist(&pre_mcmc_data._mcmc_in)) = 1 and &id. ^= %str() %then %do;
	
		%check_variables_data(variables=&id.,
													variable_text=Subject identifier,
													dataset=&pre_mcmc_data._mcmc_in,
													dataset_text=input_data,
													section=ID);
	%end;
	%else %do;
	
		%update_logger(error_code=1, section=ID, message=ID variable could not be checked for existence in input data due to errors in variable or data);
	%end;
	
	%if &repeat_obs. = %str() %then %do;
	
		%update_logger(error_code=3, section=Repeat, message=Repeat observation variable not specified);
	%end;
	%else %do;
	
		%update_logger(error_code=0, section=Repeat, message=Repeat observation variable specified);
	%end;
	
	%if %sysfunc(exist(&pre_mcmc_data._mcmc_in)) = 1 and &repeat_obs. ^= %str() %then %do;
	
		%check_variables_data(variables=&repeat_obs.,
													variable_text=Repeat observation variable,
													dataset=&pre_mcmc_data._mcmc_in,
													dataset_text=input_data,
													section=Repeat);
	%end;
	%else %do;
	
		%update_logger(error_code=1, section=Repeat, message=Repeat observation variable could not be checked for existence in input data due to errors in variable or data);
	%end;
	
	%if &weight. ^= %str() %then %do;
	
		%if %sysfunc(exist(&pre_mcmc_data._mcmc_in)) = 1 %then %do;
		
			%check_variables_data(variables=&weight.,
														variable_text=Weight variable,
														dataset=&pre_mcmc_data._mcmc_in,
														dataset_text=input_data,
														section=Weight);
		%end;
		%else %do;
		
			%update_logger(error_code=1, section=Weight, message=Weight variable could not be checked for existence in input data due to errors in input data);
		%end;
	%end;
	
	**Backtransformation;
	%if %sysfunc(exist(&pre_mcmc_data._backtran)) = 0 %then %do;
	
		%update_logger(error_code=3, section=Backtran, message=&pre_mcmc_data..backtransformation not found);
	%end;
	%else %do;
	
		%update_logger(error_code=0, section=Backtran, message=&pre_mcmc_data..backtransformation found);
	%end;
	
	%if %sysfunc(exist(&pre_mcmc_data._backtran)) = 1 %then %do;
	
		%check_variables_data(variables=variable tran_lambda minamount tran_center tran_scale biomarker,
													variable_text=Variable,
													dataset=&pre_mcmc_data._backtran,
													dataset_text=backtran,
													section=Backtran);
	%end;
	%else %do;
	
		%update_logger(error_code=1, section=Backtran, message=Variables could not be checked for existence in backtransformation data due to errors in backtransformation data);
	%end;
%mend check_data;

**Checks variable list parameters;
%macro check_variable_lists(pre_mcmc_data=,
														episodic_variables=,
														episodic_indicators=,
														episodic_amounts=,
														daily_variables=,
														daily_amounts=,
														never_consumer_variable=,
														never_consumer_indicator=,
														never_consumer_amount=);
														
	%local data_checks_passed backtran_checks_passed
				 is_nci_multivar;
														
	data _NULL_;
		set _log_record end=_last;
		
		retain _check_data _check_backtran;
		
		if _N_ = 1 then do;
		
			_check_data = 1;
			_check_backtran = 1;
		end;
		
		if section = "Data" and error_code = 3 then _check_data = 0;
		if section = "Backtran" and error_code = 3 then _check_backtran = 0;
		
		if _last = 1 then do;
		
			call symputx("data_checks_passed", _check_data);
			call symputx("backtran_checks_passed", _check_backtran);
		end;
	run;
	
	%let is_nci_multivar = 0;
	%if &data_checks_passed. = 1 %then %do;
	
		data _NULL_;
			_dsid = open("&pre_mcmc_data._mcmc_in");
			_check = (varnum(_dsid, "_nci_multivar") ^= 0);
			
			call symputx("is_nci_multivar", _check);
		run;
	%end;
	
	%if &data_checks_passed. = 0 %then %do;	
			
			%update_logger(error_code=1, section=Variables, message=nci_multivar_preprocessor usage could not be determined due to errors in input data);
	%end;
	
	**Episodic variables;
	%if &episodic_variables. ^= %str() %then %do;
	
		%if &backtran_checks_passed. = 1 %then %do;
		
			%check_variables_column(variables=&episodic_variables.,
															variable_text=Episodic variable,
															dataset=&pre_mcmc_data._backtran,
															column=variable,
															column_text=backtransformation variables,
															section=Variables);
		%end;
		%else %do;
		
			%update_logger(error_code=1, section=Variables, message=Episodic variables could not be checked for existence in the backtransformation variables due to errors in backtransformation data);
		%end;
	%end;
	
	%if &episodic_variables. ^= %str() and &episodic_indicators. = %str() %then %do;
	
		%if &is_nci_multivar. = 0 %then %do;
		
			%update_logger(error_code=3, section=Variables, message=Episodic indicators not specified with episodic variables when nci_multivar_preprocessor was not used);
		%end;
	%end;
	
	%if &episodic_variables. ^= %str() and &episodic_amounts. = %str() %then %do;
	
		%if &is_nci_multivar. = 0 %then %do;
		
			%update_logger(error_code=3, section=Variables, message=Episodic amounts not specified with episodic variables when nci_multivar_preprocessor was not used);
		%end;
	%end;
	
	%if &episodic_variables. = %str() and &episodic_indicators. ^= %str() %then %do;
	
		%if &backtran_checks_passed. = 1 %then %do;
		
			%check_backtran(variables=&episodic_indicators.,
											backtran=&pre_mcmc_data._backtran,
											section=Variables);
		%end;
		%else %do;
		
			%update_logger(error_code=1, section=Variables, message=Episodic indicators could not be checked for correspondence with backtransformation variables due to errors in backtransformation data);
		%end;
	%end;
	
	%if &episodic_variables. = %str() and &episodic_amounts. ^= %str() %then %do;
	
		%if &backtran_checks_passed. = 1 %then %do;
		
			%check_backtran(variables=&episodic_amounts.,
											backtran=&pre_mcmc_data._backtran,
											section=Variables);
		%end;
		%else %do;
		
			%update_logger(error_code=1, section=Variables, message=Episodic amounts could not be checked for correspondence with backtransformation variables due to errors in backtransformation data);
		%end;
	%end;
	
	%if &episodic_indicators. ^= %str() %then %do;
	
		%if &data_checks_passed. = 1 %then %do;
		
			%check_variables_data(variables=&episodic_indicators.,
														variable_text=Episodic indicator,
														dataset=&pre_mcmc_data._mcmc_in,
														dataset_text=input data,
														section=Variables);
		%end;
		%else %do;
		
			%update_logger(error_code=1, section=Variables, message=Episodic indicators could not be checked for existence in input data due to errors in input data);
		%end;
	%end;
	
	%if &episodic_amounts. ^= %str() %then %do;
	
		%if &data_checks_passed. = 1 %then %do;
		
			%check_variables_data(variables=&episodic_amounts.,
														variable_text=Episodic amount,
														dataset=&pre_mcmc_data._mcmc_in,
														dataset_text=input data,
														section=Variables);
		%end;
		%else %do;
		
			%update_logger(error_code=1, section=Variables, message=Episodic amounts could not be checked for existence in input data due to errors in input data);
		%end;
	%end;
	
	**Daily variables;
	%if &daily_variables. ^= %str() %then %do;
	
		%if &backtran_checks_passed. = 1 %then %do;
		
			%check_variables_column(variables=&daily_variables.,
															variable_text=Daily variable,
															dataset=&pre_mcmc_data._backtran,
															column=variable,
															column_text=backtransformation variables,
															section=Variables);
		%end;
		%else %do;
		
			%update_logger(error_code=1, section=Variables, message=Daily variables could not be checked for existence in the backtransformation variables due to errors in backtransformation data);
		%end;
	%end;
	
	%if &daily_variables. ^= %str() and &daily_amounts. = %str() %then %do;
	
		%if &is_nci_multivar. = 0 %then %do;
		
			%update_logger(error_code=3, section=Variables, message=Daily amounts not specified with daily variables when nci_multivar_preprocessor was not used);
		%end;
	%end;
	
	%if &daily_variables. = %str() and &daily_amounts. ^= %str() %then %do;
	
		%if &backtran_checks_passed. = 1 %then %do;
		
			%check_backtran(variables=&daily_amounts.,
											backtran=&pre_mcmc_data._backtran,
											section=Variables);
		%end;
		%else %do;
		
			%update_logger(error_code=1, section=Variables, message=Daily amounts could not be checked for correspondence with backtransformation variables due to errors in backtransformation data);
		%end;
	%end;
	
	%if &daily_amounts. ^= %str() %then %do;
	
		%if &data_checks_passed. = 1 %then %do;
		
			%check_variables_data(variables=&daily_amounts.,
														variable_text=Daily amount,
														dataset=&pre_mcmc_data._mcmc_in,
														dataset_text=input data,
														section=Variables);
		%end;
		%else %do;
		
			%update_logger(error_code=1, section=Variables, message=Daily amounts could not be checked for existence in input data due to errors in input data);
		%end;
	%end;
	
	**Never-consumer variable;
	%if &never_consumer_variable. ^= %str() %then %do;
	
		%if &backtran_checks_passed. = 1 %then %do;
		
			%check_variables_column(variables=&never_consumer_variable.,
															variable_text=Never-consumer variable,
															dataset=&pre_mcmc_data._backtran,
															column=variable,
															column_text=backtransformation variables,
															section=Variables);
		%end;
		%else %do;
		
			%update_logger(error_code=1, section=Variables, message=Never-consumer variable could not be checked for existence in the backtransformation variables due to errors in backtransformation data);
		%end;
	%end;
	
	%if %sysfunc(countw(&never_consumer_variable., %str( ))) > 1 %then %do;
	
		%update_logger(error_code=3, section=Variables, message=More than one never-consumer variable specified);
	%end;
	%else %do;
	
		%update_logger(error_code=0, section=Variables, message=Zero or one never-consumer variable specified);
	%end;
	
	%if &never_consumer_variable. ^= %str() and &never_consumer_indicator. = %str() %then %do;
	
		%if &is_nci_multivar. = 0 %then %do;
		
			%update_logger(error_code=3, section=Variables, message=Never-consumer indicator not specified with never-consumer variable when nci_multivar_preprocessor was not used);
		%end;
	%end;
	
	%if &never_consumer_variable. ^= %str() and &never_consumer_amount. = %str() %then %do;
	
		%if &is_nci_multivar. = 0 %then %do;
		
			%update_logger(error_code=3, section=Variables, message=Never-consumer amount not specified with never-consumer variable when nci_multivar_preprocessor was not used);
		%end;
	%end;
	
	%if &never_consumer_variable. = %str() and &never_consumer_indicator. ^= %str() %then %do;
	
		%if &backtran_checks_passed. = 1 %then %do;
		
			%check_backtran(variables=&never_consumer_indicator.,
											backtran=&pre_mcmc_data._backtran,
											section=Variables);
		%end;
		%else %do;
		
			%update_logger(error_code=1, section=Variables, message=Never-consumer indicator could not be checked for correspondence with backtransformation variables due to errors in backtransformation data);
		%end;
	%end;
	
	%if &never_consumer_variable. = %str() and &never_consumer_amount. ^= %str() %then %do;
	
		%if &backtran_checks_passed. = 1 %then %do;
		
			%check_backtran(variables=&never_consumer_amount.,
											backtran=&pre_mcmc_data._backtran,
											section=Variables);
		%end;
		%else %do;
		
			%update_logger(error_code=1, section=Variables, message=Never-consumer amount could not be checked for correspondence with backtransformation variables due to errors in backtransformation data);
		%end;
	%end;
	
	%if &never_consumer_indicator. ^= %str() %then %do;
	
		%if &data_checks_passed. = 1 %then %do;
		
			%check_variables_data(variables=&never_consumer_indicator.,
														variable_text=Never-consumer indicator,
														dataset=&pre_mcmc_data._mcmc_in,
														dataset_text=input data,
														section=Variables);
		%end;
		%else %do;
		
			%update_logger(error_code=1, section=Variables, message=Never-consumer indicator could not be checked for existence in input data due to errors in input data);
		%end;
	%end;
	
	%if %sysfunc(countw(&never_consumer_indicator., %str( ))) > 1 %then %do;
	
		%update_logger(error_code=3, section=Variables, message=More than one never-consumer indicator specified);
	%end;
	%else %do;
	
		%update_logger(error_code=0, section=Variables, message=Zero or one never-consumer indicator specified);
	%end;
	
	%if &never_consumer_amount. ^= %str() %then %do;
	
		%if &data_checks_passed. = 1 %then %do;
		
			%check_variables_data(variables=&never_consumer_amount.,
														variable_text=Never-consumer amount,
														dataset=&pre_mcmc_data._mcmc_in,
														dataset_text=input data,
														section=Variables);
		%end;
		%else %do;
		
			%update_logger(error_code=1, section=Variables, message=Never-consumer amount could not be checked for existence in input data due to errors in input data);
		%end;
	%end;
	
	%if %sysfunc(countw(&never_consumer_amount., %str( ))) > 1 %then %do;
	
		%update_logger(error_code=3, section=Variables, message=More than one never-consumer amount specified);
	%end;
	%else %do;
	
		%update_logger(error_code=0, section=Variables, message=Zero or one never-consumer amount specified);
	%end;
%mend check_variable_lists;

**Checks covariate list parameters;
%macro check_covariate_lists(pre_mcmc_data=,
														 default_covariates=,
														 episodic_indicator_covariates=,
														 episodic_amount_covariates=,
														 daily_amount_covariates=,
														 individual_covariates=,
														 never_consumer_covariates=,
														 episodic_variables=,
														 episodic_indicators=,
														 episodic_amounts=,
														 daily_variables=,
														 daily_amounts=,
														 never_consumer_variable=,
														 never_consumer_indicator=,
														 never_consumer_amount=);
														 
	%local data_checks_passed variable_checks_passed
				 is_nci_multivar;
														 
	data _NULL_;
		set _log_record end=_last;
		
		retain _check_data _check_variables;
		
		if _N_ = 1 then do;
		
			_check_data = 1;
			_check_variables = 1;
		end;
		
		if section = "Data" and error_code = 3 then _check_data = 0;
		if section = "Variables" and error_code = 3 then _check_variables = 0;
		
		if _last = 1 then do;
		
			call symputx("data_checks_passed", _check_data);
			call symputx("variable_checks_passed", _check_variables);
		end;
	run;
	
	%let is_nci_multivar = 0;
	%if &data_checks_passed. = 1 %then %do;
	
		data _NULL_;
			_dsid = open("&pre_mcmc_data._mcmc_in");
			_check = (varnum(_dsid, "_nci_multivar") ^= 0);
			
			call symputx("is_nci_multivar", _check);
		run;
	%end;
	
	%if &default_covariates. ^= %str() %then %do;
	
		%if &data_checks_passed. = 1 %then %do;
		
			%check_variables_data(variables=&default_covariates.,
														variable_text=Default covariate,
														dataset=&pre_mcmc_data._mcmc_in,
														dataset_text=input data,
														section=Covariates);
		%end;
		%else %do;
		
			%update_logger(error_code=1, section=Covariates, message=Default covariate existence in input data cannot be checked due to errors in input data);
		%end;
	%end;
	
	%if &episodic_indicator_covariates. ^= %str() %then %do;
	
		%if &data_checks_passed. = 1 %then %do;
		
			%check_variables_data(variables=&episodic_indicator_covariates.,
														variable_text=Episodic indicator covariate,
														dataset=&pre_mcmc_data._mcmc_in,
														dataset_text=input data,
														section=Covariates);
		%end;
		%else %do;
		
			%update_logger(error_code=1, section=Covariates, message=Episodic indicator covariate existence in input data cannot be checked due to errors in input data);
		%end;
	%end;
	
	%if &episodic_amount_covariates. ^= %str() %then %do;
	
		%if &data_checks_passed. = 1 %then %do;
		
			%check_variables_data(variables=&episodic_amount_covariates.,
														variable_text=Episodic amount covariate,
														dataset=&pre_mcmc_data._mcmc_in,
														dataset_text=input data,
														section=Covariates);
		%end;
		%else %do;
		
			%update_logger(error_code=1, section=Covariates, message=Episodic amount covariate existence in input data cannot be checked due to errors in input data);
		%end;
	%end;
	
	%if &daily_amount_covariates. ^= %str() %then %do;
	
		%if &data_checks_passed. = 1 %then %do;
		
			%check_variables_data(variables=&daily_amount_covariates.,
														variable_text=Daily amount covariate,
														dataset=&pre_mcmc_data._mcmc_in,
														dataset_text=input data,
														section=Covariates);
		%end;
		%else %do;
		
			%update_logger(error_code=1, section=Covariates, message=Daily amount covariate existence in input data cannot be checked due to errors in input data);
		%end;
	%end;
	
	%if &individual_covariates. ^= %str() %then %do;
	
		%local indiv_covar_checks_passed;
	
		%check_variables_data(variables=variable covariates,
													variable_text=Column,
													dataset=&individual_covariates.,
													dataset_text=individual covariate data,
													section=Individual Covariate Data);
				
		%let indiv_covar_checks_passed = 1;									
		data _NULL_;
			set _log_record;
			where section = "Individual Covariate Data";
			
			if error_code = 3 then call symputx("indiv_covar_checks_passed", 0);
		run;
		
		%if &indiv_covar_checks_passed. = 1 and &variable_checks_passed. = 1 %then %do;
		
			%local indiv_covariate_names;
		
			**process indicator and amount lists;
			%process_variable_lists(pre_mcmc_data=&pre_mcmc_data.,
															episodic_variables=&episodic_variables.,
															episodic_indicators=&episodic_indicators.,
															episodic_amounts=&episodic_amounts.,
															daily_variables=&daily_variables.,
															daily_amounts=&daily_amounts.,
															never_consumer_variable=&never_consumer_variable.,
															never_consumer_indicator=&never_consumer_indicator.,
															never_consumer_amount=&never_consumer_amount.);
															
			data _NULL_;
				set _vars;
		
				array episodic_ind{*} episodic_ind:;
				array episodic_amt{*} episodic_amt:;
				array daily_amt{*} daily_amt:;
		
				episodic_indicators = catx(" ", of episodic_ind{*});
				episodic_amounts = catx(" ", of episodic_amt{*});
				daily_amounts = catx(" ", of daily_amt{*});
		
				call symputx("episodic_indicators", episodic_indicators);
				call symputx("episodic_amounts", episodic_amounts);
				call symputx("daily_amounts", daily_amounts);
			run;
		
			data _NULL_;
				set &individual_covariates. end=last;
				
				length indiv_covariate_names $32767.;
				
				retain indiv_covariate_names;
				
				if _N_ = 1 then indiv_covariate_names = "";
				
				indiv_covariate_names = catx(" ", indiv_covariate_names, variable);
				
				if last = 1 then call symputx("indiv_covariate_names", indiv_covariate_names);
			run;
		
			%check_variables_macro_list(variables=&indiv_covariate_names.,
																	variable_text=Individual covariate list name,
																	list=&episodic_indicators. &episodic_amounts. &daily_amounts.,
																	list_text=indicators and amounts,
																	section=Covariates);
		%end;
		%else %do;
		
			%update_logger(error_code=1, section=Covariates, message=Individual covariate list names cannot be checked because individual covariate data or variable lists are not specified correctly);
		%end;
		
		%if &indiv_covar_checks_passed. = 1 and &data_checks_passed. = 1 %then %do;
		
			%local num_covariate_lists i covariates variable;
		
			data _NULL_;
				set &individual_covariates. end=last;
				
				if last = 1 then call symputx("num_covariate_lists", _N_);
			run;
			
			%do i = 1 %to &num_covariate_lists.;
			
				data _NULL_;
					set &individual_covariates.;
					
					if _N_ = &i. then do;
					
						call symputx("covariates", covariates);
						call symputx("variable", variable);
					end;
				run;
				
				%check_variables_data(variables=&covariates.,
															variable_text=&variable. covariate,
															dataset=&pre_mcmc_data._mcmc_in,
															dataset_text=input data,
															section=Covariates);
			%end;
		%end;
		%else %do;
		
			%update_logger(error_code=1, section=Covariates, message=Individual covariates could not be checked for existence in input data because individual covariate data or input data is not specified correctly);
		%end;
	%end;
	
	%if &never_consumer_covariates. ^= %str() %then %do;
	
		%if &data_checks_passed. = 1 %then %do;
		
			%check_variables_data(variables=&never_consumer_covariates.,
														variable_text=Never-consumer covariate,
														dataset=&pre_mcmc_data._mcmc_in,
														dataset_text=input data,
														section=Covariates);
		%end;
	%end;
%mend check_covariate_lists;

%macro check_sigma_u_prior(sigma_u_prior=,
													 pre_mcmc_data=,
													 episodic_variables=,
													 episodic_indicators=,
													 episodic_amounts=,
													 daily_variables=,
													 daily_amounts=,
													 never_consumer_variable=,
													 never_consumer_indicator=,
													 never_consumer_amount=);
													 
	%if &sigma_u_prior. ^= %str() %then %do;
	
		%local variable_checks_passed;
	
		data _NULL_;
			set _log_record end=_last;
		
			retain _check_data _check_variables;
		
			if _N_ = 1 then do;
		
				_check_variables = 1;
			end;
		
			if section = "Variables" and error_code = 3 then _check_variables = 0;
		
			if _last = 1 then do;
		
				call symputx("variable_checks_passed", _check_variables);
			end;
		run;
		
		%if &variable_checks_passed. = 1 %then %do;
		
			%process_variable_lists(pre_mcmc_data=&pre_mcmc_data.,
															episodic_variables=&episodic_variables.,
															episodic_indicators=&episodic_indicators.,
															episodic_amounts=&episodic_amounts.,
															daily_variables=&daily_variables.,
															daily_amounts=&daily_amounts.,
															never_consumer_variable=&never_consumer_variable.,
															never_consumer_indicator=&never_consumer_indicator.,
															never_consumer_amount=&never_consumer_amount.);
		%end;
		
		proc iml;
		
			**Load IML logger module;
			load module=update_logger;
			
			use &sigma_u_prior.;
				read all var _ALL_ into _sigma_u_prior;
			close &sigma_u_prior.;
			
			_variable_checks_passed = &variable_checks_passed.;
			
			**Check if Sigma-u prior is square;
			if nrow(_sigma_u_prior) = ncol(_sigma_u_prior) then square = 1;
			else square = 0;
			
			if square = 0 then do;
				
				call update_logger(3, "Priors", "Sigma-u prior is not square");
			end;
			else do;
			
				call update_logger(0, "Priors", "Sigma-u prior is square");
			end;
			
			**Check if size of Sigma-u prior matches number of indicators and amounts;
			if square = 1 then do;
			
				if _variable_checks_passed = 1 then do;
				
					use _vars;
					read all var {episodic_ind:} into episodic_indicators;
					read all var {episodic_amt:} into episodic_amounts;
					read all var {daily_amt:} into daily_amounts;
					close _vars;
					num_variables = ncol(episodic_indicators) + ncol(episodic_amounts) + ncol(daily_amounts);
				
					if nrow(_sigma_u_prior) ^= num_variables then do;
				
						call update_logger(3, "Priors", cat("Size of Sigma-u prior (", nrow(sigma_u_prior), ") does not match number of variables (", num_variables, ")"));
					end;
					else do;
				
						call update_logger(0, "Priors", cat("Size of Sigma-u prior matches number of variables"));
					end;
				end;
				else do;
				
					call update_logger(1, "Priors", "Sigma-u prior size could not be checked against number of variables because variables were not specified correctly");
				end;
			end;
			else do;
			
				call update_logger(1, "Priors", "Sigma-u prior size could not be checked because Sigma-u prior is not square");
			end;
			
			**Check if Sigma-u prior is symmetric;
			if square = 1 then do;
			
				if all(abs(_sigma_u_prior - _sigma_u_prior`) < 10**-8) then symmetric = 1;
				else symmetric = 0;
			end;
			else symmetric = 0;
			
			if symmetric = 0 then do;
			
				call update_logger(3, "Priors", "Sigma-u prior is not symmetric");
			end;
			else do;
			
				call update_logger(0, "Priors", "Sigma-u prior is symmetric");
			end;
			
			**Check if Sigma-u prior is positive-definite;
			if symmetric = 1 then do;
			
				call eigen(eigvals, eigvecs, _sigma_u_prior);
				if any(eigvals < 10**-8) then do;
				
					call update_logger(3, "Priors", "Sigma-u prior is not positive-definite");
				end;
				else do;
				
					call update_logger(0, "Priors", "Sigma-u prior is positive-definite");
				end;
			end;
		quit;
	%end;
%mend check_sigma_u_prior;

**Checks input parameters to the MCMC function;
%macro check_input(num_mcmc_iterations=,
									 num_burn=,
									 num_thin=,
									 num_post=,
									 pre_mcmc_data=,
									 id=,
									 repeat_obs=,
									 weight=,
									 episodic_variables=,
									 episodic_indicators=,
									 episodic_amounts=,
									 daily_variables=,
									 daily_amounts=,
									 never_consumer_variable=,
									 never_consumer_indicator=,
									 never_consumer_amount=,
									 default_covariates=,
									 episodic_indicator_covariates=,
									 episodic_amount_covariates=,
									 daily_amount_covariates=,
									 individual_covariates=,
									 never_consumer_covariates=,
									 sigma_u_prior=);

	%check_iterations(num_mcmc_iterations=&num_mcmc_iterations.,
										num_burn=&num_burn.,
										num_thin=&num_thin.,
										num_post=&num_post.);
										
	%check_data(pre_mcmc_data=&pre_mcmc_data.,
							id=&id.,
							repeat_obs=&repeat_obs.,
							weight=&weight.);
							
	%check_variable_lists(pre_mcmc_data=&pre_mcmc_data.,
												episodic_variables=&episodic_variables.,
												episodic_indicators=&episodic_indicators.,
												episodic_amounts=&episodic_amounts.,
												daily_variables=&daily_variables.,
												daily_amounts=&daily_amounts.,
												never_consumer_variable=&never_consumer_variable.,
												never_consumer_indicator=&never_consumer_indicator.,
												never_consumer_amount=&never_consumer_amount.);
												
	%check_covariate_lists(pre_mcmc_data=&pre_mcmc_data.,
												 default_covariates=&default_covariates.,
												 episodic_indicator_covariates=&episodic_indicator_covariates.,
												 episodic_amount_covariates=&episodic_amount_covariates.,
												 daily_amount_covariates=&daily_amount_covariates.,
												 individual_covariates=&individual_covariates.,
												 never_consumer_covariates=&never_consumer_covariates.,
												 episodic_variables=&episodic_variables.,
												 episodic_indicators=&episodic_indicators.,
												 episodic_amounts=&episodic_amounts.,
												 daily_variables=&daily_variables.,
												 daily_amounts=&daily_amounts.,
												 never_consumer_variable=&never_consumer_variable.,
												 never_consumer_indicator=&never_consumer_indicator.,
												 never_consumer_amount=&never_consumer_amount.);
												 
	%check_sigma_u_prior(sigma_u_prior=&sigma_u_prior.,
											 pre_mcmc_data=&pre_mcmc_data.,
											 episodic_variables=&episodic_variables.,
											 episodic_indicators=&episodic_indicators.,
											 episodic_amounts=&episodic_amounts.,
											 daily_variables=&daily_variables.,
											 daily_amounts=&daily_amounts.,
											 never_consumer_variable=&never_consumer_variable.,
											 never_consumer_indicator=&never_consumer_indicator.,
											 never_consumer_amount=&never_consumer_amount.);
												 
	data _NULL_;
		set _log_record;
		
		if error_code = 3 then abort abend 2;
	run;
%mend check_input;