/*
Program Purpose:
IML modules for estimating MCMC log-likelihood
*/

libname _modules ".";

proc iml;
	
	**Multivariate normal cumulative distribution function;
	start multi_norm_cdf(x,
											 sigma,
											 Nmax=0,
											 tol=0.001);
											 
		m = nrow(x);
		
		**Special cases for univariate and bivariate distributions;					 
		if m = 1 then do;
		
			result = cdf("Normal", x);
			return result;
		end;
		
		if m = 2 then do;
			
			rho = sigma[1,2]/sqrt(sigma[1,1]*sigma[2,2]);
			result = probbnrm(x[1], x[2], rho);
			return result;
		end;
		
		**Set default maximum number of iterations if not provided;
		if Nmax = 0 then Nmax = nrow(x)*50000;
		
		**Reorder x and compute Cholesky factor;
		call sort(x);
		C = root(x);
		
		**Monte Carlo sampler from Genz (1992);
		d_init = 0;
		e_init = cdf("Normal", x[1]/C[1,1]);
		f_init = e_init - d_init;
		
		result = 0;
		sum_sq_diff = 0;
		variance = 0;
		error = 0;
		N = 1;
		do until(N > Nmax | error <= tol);
		
			d = d_init;
			e = e_init;
			f = f_init;
			
			w = j(m - 1, 1, 0);
			call randgen(w, "Uniform");
			y = j(m - 1, 1, 0);
			
			do j = 2 to m;
			
				y[j-1] = quantile("Normal", d + w[j-1]*(e-d));
				d = 0;
				e = cdf("Normal", x[j] - C[j,1:(j-1)]*y[1:(j-1)]);
				f = (e - d)*f;
			end;
			
			**Update mean and variance;
			delta1 = f - result;
			result = result + delta1/N;
			delta2 = f - result;
			sum_sq_diff = sum_sq_diff + delta1*delta2;
			
			**Calculate error;
			**Loop will terminate once desired accuracy is reached;
			if N >= 2 then do;
			
				variance = sum_sq_diff/(N - 1);
				error = 3*sqrt(variance/N);
			end;
			
			N = N + 1;
		end;
		
		return result;
	finish;
	
	**Joint log-likelihood of observed parameters and random effects;
	**This is the likelihood from Appendix A.4 from Zhang, et al. (2011) with latent W matrix elements integrated out;
	**For never-consumer models, latent Ni variables are integrated out as in Bhadra, et al. (2020);
	start joint_likelihood(u_matrix_i)
				global(_w_matrix_i,
							 _episodic_indicator_i,
							 _xbeta_i,
							 _sigma_u,
							 _sigma_e,
							 _recall_availability_i,
							 _num_episodic,
							 _num_daily,
							 _num_recalls,
							 _has_never_consumers,
							 _g_i,
							 _alpha1);
												 
		pi = arcos(-1);
												 
		num_variables = 2*_num_episodic + _num_daily;
		
		**Indices of episodic indicators, episodic amounts, and daily amounts;
		if _num_episodic > 0 then do;
		
			ep_ind = do(1, 2*_num_episodic, 2);
			ep_amt = do(2, 2*_num_episodic, 2);
		end;
		
		if _num_daily > 0 then do;
		
			dy_amt = do(2*_num_episodic + 1, 2*_num_episodic + _num_daily, 1);
		end;

		ll_prob1 = 0;
		ll_prob2 = 0;
		ll_amt = 0;
		
		if _has_never_consumers = 1 then do;
		
			num_consumption = 0;
			do day_k = 1 to _num_recalls;
			
				if _recall_availability_i[day_k] = 1 then do;
				
					episodic_indicator_i_k = _episodic_indicator_i$day_k;
					num_consumption = num_consumption + episodic_indicator_i_k[1];
				end;
			end;
			
			g_alpha = _g_i * _alpha1;
			
			ll_prob1 = ll_prob1 + log(max(cdf("Normal", g_alpha), 0.00000001));
			if num_consumption = 0 then do;
			
				ll_prob2 = ll_prob2 + log(max(cdf("Normal", -g_alpha), 0.00000001));
			end;
		end;
		
		do day_k = 1 to _num_recalls;
		
			if _recall_availability_i[day_k] = 1 then do;
			
				xbeta_u = _xbeta_i$day_k + u_matrix_i;
				w_minus_xbeta_u = _w_matrix_i$day_k - xbeta_u;
				
				observed = j(num_variables, 1, .);
				if _num_episodic > 0 then do;
				
					observed[ep_ind] = 0;
					observed[ep_amt] = _episodic_indicator_i$day_k;
				end;
				
				if _num_daily > 0 then do;
				
					observed[dy_amt] = 1;
				end;
				
				num_observed = sum(observed);
				
				**Log-likelihood for probability part of model;
				if _num_episodic > 0 then do;
				
					**Split multivariate normal distribution of W-XBeta-U into probability and amount parts using the Schur complement, integrating out latent W variables;
					**Unobserved amounts are simply dropped because they will integrate to 1;
					xbeta_u_prob = xbeta_u[ep_ind];
					sigma_e_prob = _sigma_e[ep_ind, ep_ind];
					if num_observed > 0 then do;
					
						xbeta_u_prob = xbeta_u_prob + w_minus_xbeta_u[observed] * inv(_sigma_e[observed, observed]) * _sigma_e[observed, ep_ind];
						sigma_e_prob = sigma_e_prob - _sigma_e[ep_ind, observed] * inv(_sigma_e[observed, observed]) * _sigma_e[observed, ep_ind];
					end;
					
					prob1 = multi_norm_cdf((2 # _episodic_indicator_i$day_k - 1)*xbeta_u_prob, sigma_e_prob);
					ll_prob1 = ll_prob1 + log(max(prob1, 0.00000001));
					
					if _has_never_consumers = 1 then do;
					
						if num_consumption = 0 then do;
						
							prob2 = prob1/cdf("Normal", -xbeta_u_prob[1]);
							ll_prob2 = ll_prob2 + log(max(prob2, 0.00000001));
						end;
					end;
				end;
				
				**Log-likelihood for amount part of model;
				if num_observed > 0 then do;
				
					w_minus_xbeta_u_amt = w_minus_xbeta_u[observed];
					sigma_e_amt = _sigma_e[observed, observed];
					
					ll_amt = ll_amt - 0.5*num_observed*log(2*pi) - 0.5*log(det(sigma_e_amt)) - 0.5*(w_minus_xbeta_u_amt * inv(sigma_e_amt) * w_minus_xbeta_u_amt`);
				end;
			end;
		end;
		
		if _has_never_consumers = 1 then do;
		
			if num_consumption = 0 then do;
			
				**Log-Sum-Exp trick to add consumer and never-consumer probabilities for potential never-consumers;
				ll_prob_min = min(ll_prob1, ll_prob2);
				ll_prob1 = ll_prob1 - ll_prob_min;
				ll_prob2 = ll_prob2 - ll_prob_min;
				
				ll_prob = log(exp(ll_prob1) + exp(ll_prob2)) + ll_prob_min;
			end;
			else do;
			
				ll_prob = ll_prob1;
			end;
		end;
		else do;
		
			ll_prob = ll_prob1;
		end;
		
		ll = ll_prob + ll_amt;
		
		**Log-likelihood of random effects;
		ll = ll - 0.5*num_variables*log(2*pi) - 0.5*log(det(_sigma_u)) - 0.5*(u_matrix_i * inv(_sigma_u) * u_matrix_i`);
		
		return ll;
	finish;
	
	start marginal_likelihood(do_log_likelihood,
														w_matrix,
														episodic_indicator_matrices,
														xbeta,
														sigma_u,
														sigma_e,
														recall_availability,
														subject_weighting,
														num_subjects,
														num_episodic,
														num_daily,
														num_recalls,
														has_never_consumers,
														never_consumer_covariate_matrix,
														alpha1)
				global(_w_matrix_i,
							 _episodic_indicator_i,
							 _xbeta_i,
							 _sigma_u,
							 _sigma_e,
							 _recall_availability_i,
							 _num_episodic,
							 _num_daily,
							 _num_recalls,
							 _has_never_consumers,
							 _g_i,
							 _alpha1);
														
		if upcase(do_log_likelihood) ^= "Y" then do;
		
			return .;
		end;
		
		pi = arcos(-1);
		
		num_variables = 2*num_episodic + num_daily;
		
		**Set precision level in digits of algorithm;
		if num_episodic > 2 then precision = 3; **For more than two episodic variables, precision is bounded by the Monte Carlo algorithm used for the multivariate normal CDF;
		else precision = 8;
		
		**Set option and control lists for optimizer call;
		optimizer_options = j(11, 1, .);
		optimizer_options[1] = 1; **Maximization;
		optimizer_options[4] = 3; **BFGS algorithm;
		optimizer_options[8] = 01; **Central difference for derivatives, interval calculated using only algorithm precision;
		
		optimizer_control = j(10, 1, .);
		optimizer_control[8] = precision; **Algorithm precision;
		
		**Set control list for derivative evaluation call;
		derivative_control = j(3, 1, .);
		derivative_control[2] = 1; **Central difference for derivatives, interval calculated using only algorithm precision;
		derivative_control[3] = precision; **Algorithm precision;
		
		**Find joint log-likelihood at posterior mode of U matrix;
		ll = j(num_subjects, 1, 0);
		log_det_hessian = j(num_subjects, 1, .);
		do i = 1 to num_subjects;
		
			**Define variables for log-likelihood function;
			_w_matrix_i = ListCreate(num_recalls);
			_episodic_indicator_i = ListCreate(num_recalls);
			_xbeta_i = ListCreate(num_recalls);
			do day_k = 1 to num_recalls;
			
				w_matrix_k = w_matrix$day_k;
				_w_matrix_i$day_k = w_matrix_k[i,];
				
				if num_episodic > 0 then do;
				
					episodic_indicator_k = episodic_indicator_matrices$day_k;
					_episodic_indicator_i$day_k = episodic_indicator_k[i,];
				end;
				else do;
				
					_episodic_indicator_i$day_k = {};
				end;
				
				xbeta_k = xbeta$day_k;
				_xbeta_i$day_k = xbeta_k[i,];
			end;
			
			_sigma_u = sigma_u;
		
			_sigma_e = sigma_e;
			
			_recall_availability_i = recall_availability[i,];
			
			_num_episodic = num_episodic;
			_num_daily = num_daily;
			_num_recalls = num_recalls;
			
			_has_never_consumers = has_never_consumers;
			if _has_never_consumers = 1 then do;
			
				_g_i = never_consumer_covariate_matrix[i,];
				_alpha1 = alpha1;
			end;
			else do;
			
				_g_i = {};
				_alpha1 = {};
			end;
			
			**Find value and Hessian of Ui at posterior mode;
			u0 = j(num_variables, 1, 0);
			call nlpqn(rc, ui_opt, "joint_likelihood", u0, optimizer_options, , , optimizer_control);
			call nlpfdd(ll_opt, ll_grad, ll_hess, "joint_likelihood", ui_opt, derivative_control);
			
			ll[i] = ll_opt;
			log_det_hessian[i] = log(det(-ll_hess));
		end;
		
		**Estimate marginal log-likelihood using Laplace approximation;
		ll = ll + 0.5*num_variables*log(2*pi) - 0.5 # log_det_hessian;
		
		**Return weighted sum of marginal log-likelihood;
		ll_total = sum(ll # subject_weighting);
		return ll_total;
	finish;
	
	reset storage = _modules.mcmc_modules;
	store module=(multi_norm_cdf
								joint_likelihood
								marginal_likelihood);
quit;