# Generates a vector of truncated normal distribution values given a vector of truncation values using rejection sampler (Robert, 1995).
# Paper source: Robert (1995). Simulation of truncated normal variables. Statistics and Computing, 5, 121-125. https://doi.org/10.48550/arXiv.0907.4010
truncated_normal <- function(trunc.values) {

  #number of truncation values
  num.trunc.values <- length(trunc.values)

  #positive and negative truncation values
  positive <- trunc.values >= 0
  negative <- trunc.values < 0

  #initialize full truncated normal result vector
  trunc.normal.values <- numeric(num.trunc.values)

  #initialize full vector of accepted values
  accepted <- logical(num.trunc.values)

  #use exponential rejection sampler for any positive truncation values
  if(any(positive)) {

    #subset positive truncation values
    trunc.values.positive <- trunc.values[positive]

    #scale factor for translated exponential distribution (used for positive truncation values)
    exp.scale.factor <- (trunc.values.positive + sqrt(4+(trunc.values.positive^2)))/2

    #initialize result vector of truncated normal values for positive truncation values
    trunc.normal.positive <- trunc.normal.values[positive]

    #initialize vector of accepted values for positive truncation values
    accepted.positive <- accepted[positive]

    #exponential rejection sampler for positive truncation values
    while(!all(accepted.positive)) {

      #determine which values to replace
      values.to.replace.positive <- !accepted.positive
      num.to.replace.positive <- sum(values.to.replace.positive)

      #calculate candidate values
      candidate.values <- trunc.values.positive[values.to.replace.positive] - ((1/exp.scale.factor[values.to.replace.positive])*log(runif(num.to.replace.positive)))

      #calculate acceptance probabilities for the candidate values and either accept or reject them
      acceptance.probability <- exp(-0.5*(candidate.values - exp.scale.factor[values.to.replace.positive])^2)
      accepted.positive[values.to.replace.positive] <- runif(sum(values.to.replace.positive)) < acceptance.probability

      trunc.normal.positive[values.to.replace.positive] <- candidate.values
    }

    trunc.normal.values[positive] <- trunc.normal.positive
  }

  #use normal rejection sampler for any negative truncation values
  if(any(negative)) {

    #subset negative truncation values
    trunc.values.negative <- trunc.values[negative]

    #initialize result vector of truncated normal values for negative truncation values
    trunc.normal.negative <- trunc.normal.values[negative]

    #initialize vector of accepted values for positive truncation values
    accepted.negative <- accepted[negative]

    #normal rejection sampler for negative truncation values
    while(!all(accepted.negative)) {

      #determine which values to replace
      values.to.replace.negative <- !accepted.negative
      num.to.replace.negative <- sum(values.to.replace.negative)

      #draw from a normal distribution with a mean of 0 and a standard deviation of 1
      new.truncated.normal.negative <- rnorm(num.to.replace.negative)
      accepted.negative[values.to.replace.negative] <- new.truncated.normal.negative >= trunc.values.negative[values.to.replace.negative]

      trunc.normal.negative[values.to.replace.negative] <- new.truncated.normal.negative
    }

    trunc.normal.values[negative] <- trunc.normal.negative
  }

  return(trunc.normal.values)
}
