#Calculates bivariate normal CDF.
#Equation (31), p. 97, chapter 36, Continuous Multivariate Normal Distributions, Johnson & Kotz.
bvpnorm <- function(x, sigma) {

  #Scale input by standard deviations
  x.scl <- t(t(x)/sqrt(diag(sigma)))

  #Univariate probabilities for each variable
  phi1 <- pnorm(x.scl[,1])
  phi2 <- pnorm(x.scl[,2])

  #Correlation between the variables
  corr <- cov2cor(sigma)
  rho <- corr[1,2]

  #Special case where variables are independent
  if(rho == 0) {

    return(phi1*phi2)
  }

  s <- sqrt(1 - rho^2)
  pi <- acos(-1)

  #calculate v1 = v(x1, (x2-rho*x1)/s)
  h1 <- x.scl[,1]
  k1 <- (x.scl[,2] - rho*x.scl[,1])/s

  abs.h1 <- abs(h1)
  abs.k1 <- abs(k1)

  h2 <- pmax(abs.h1, abs.k1)
  k2 <- pmin(abs.h1, abs.k1)
  l <- ifelse(h2 > 0, k2/h2, 0)
  m <- (h2^2)/2
  enm <- exp(-m)

  b0 <- 1 - enm
  b1 <- b0 - m*enm
  b2 <- b1 - ((m^2)/2)*enm
  b3 <- b2 - ((m^3)/6)*enm
  b4 <- b3 - ((m^4)/24)*enm

  v1 <- (1/(2*pi))*(b0*l - b1*((l^3)/3) + b2*((l^5)/5) - b3*((l^7)/7) + b4*((l^9)/9))
  v1 <- ifelse(abs.k1 > abs.h1, (pnorm(h2) - 0.5)*(pnorm(k2) - 0.5) - v1, v1)
  v1 <- ifelse(h1*k1 < 0, -v1, v1)

  #calculate v2 = v(x2, (x1-rho*x2)/s)
  h1 <- x.scl[,2]
  k1 <- (x.scl[,1] - rho*x.scl[,2])/s

  abs.h1 <- abs(h1)
  abs.k1 <- abs(k1)

  h2 <- pmax(abs.h1, abs.k1)
  k2 <- pmin(abs.h1, abs.k1)
  l <- ifelse(h2 > 0, k2/h2, 0)
  m <- (h2^2)/2
  enm <- exp(-m)

  b0 <- 1 - enm
  b1 <- b0 - m*enm
  b2 <- b1 - ((m^2)/2)*enm
  b3 <- b2 - ((m^3)/6)*enm
  b4 <- b3 - ((m^4)/24)*enm

  v2 <- (1/(2*pi))*(b0*l - b1*((l^3)/3) + b2*((l^5)/5) - b3*((l^7)/7) + b4*((l^9)/9))
  v2 <- ifelse(abs.k1 > abs.h1, (pnorm(h2) - 0.5)*(pnorm(k2) - 0.5) - v2, v2)
  v2 <- ifelse(h1*k1 < 0, -v2, v2)

  #calculate bivariate normal CDF
  phi <- v1 + v2 + (phi1 + phi2)/2 - acos(rho)/(2*pi)
  return(phi)
}

#Calculates multivariate normal CDF.
mvpnorm <- function(x, sigma, Nmax=NULL, tol=0.001) {

  if(is.vector(x)) {

    x <- t(as.matrix(x))
  }

  if(is.vector(sigma)) {

    sigma <- as.matrix(sigma)
  }

  #Use built-in standard normal CDF to get exact solution for univariate case
  if(ncol(x) == 1) {

    return(pnorm(x, sd=sqrt(sigma[1,1])))
  }

  #Special case for bivariate normal
  if(ncol(x) == 2) {

    return(bvpnorm(x, sigma=sigma))
  }

  #Set default maximum number of iterations if not provided
  if(is.null(Nmax)) {

    Nmax <- 50000*ncol(x)
  }

  #Re-order columns of x and compute Cholesky factors
  C <- array(dim=c(nrow(sigma), ncol(sigma), nrow(x)))
  for(i in 1:nrow(x)) {

    x.order <- order(x[i,])
    x[i,] <- x[i,x.order]
    C[,,i] <- t(chol(sigma[x.order,x.order]))
  }

  m <- ncol(x)

  #Monte Carlo sampler from Genz (1992)
  d.init <- numeric(nrow(x))
  e.init <- pnorm(x[,1]/C[1,1,])
  f.init <- e.init - d.init

  result <- numeric(nrow(x))
  sum.sq.diff <- numeric(nrow(x))
  variance <- numeric(nrow(x))
  error <- numeric(nrow(x))
  done <- logical(nrow(x))
  for(N in 1:Nmax) {

    d <- d.init[!done]
    e <- e.init[!done]
    f <- f.init[!done]

    w <- matrix(runif(sum(!done)*(m-1)), nrow=sum(!done), ncol=m-1)
    y <- matrix(nrow=sum(!done), ncol=m-1)

    for(j in 2:m) {

      y[,j-1] <- qnorm(d + w[,j-1]*(e - d))
      d <- numeric(sum(!done))
      e <- pnorm((x[!done,j] - rowSums(matrix(C[j,1:(j-1),!done], nrow=sum(!done), ncol=j-1, byrow=TRUE)*matrix(y[,1:(j-1)], nrow=sum(!done), ncol=j-1, byrow=TRUE)))/C[j,j,!done])
      f <- (e - d)*f
    }

    #Update mean and variance
    delta1 <- f - result[!done]
    result[!done] <- result[!done] + delta1/N
    delta2 <- f - result[!done]
    sum.sq.diff[!done] <- sum.sq.diff[!done] + delta1*delta2

    #Check error and terminate if desired accuracy is reached
    if(N >= 2) {

      variance[!done] <- sum.sq.diff[!done]/(N-1)
      error[!done] <- 3*sqrt(variance[!done]/N)
      done <- (error <= tol)

      if(all(done)) {

        break
      }
    }
  }

  attr(result, "error") <- max(error)
  attr(result, "iterations") <- N
  attr(result, "accuracy.reached") <-  all(done)
  return(result)
}

#Computes joint log-likelihood of observed parameters and random effects
#This is the likelihood from Appendix A.4 from Zhang, et al. (2011) with latent W matrix elements integrated out.
#For never-consumer models, latent Ni variables are integrated out as in Bhadra, et al. (2020).
joint_likelihood <- function(u.matrix.i,
                             w.matrix.i,
                             episodic.indicator.i,
                             xbeta.i,
                             sigma.u,
                             sigma.e,
                             recall.availability.i,
                             num.episodic,
                             num.daily,
                             num.recalls,
                             never.consumers.first.episodic,
                             never.consumer.covariate.matrix.i,
                             alpha1) {

  num.variables <- 2*num.episodic + num.daily

  #Indices of episodic indicators, episodic amounts, and daily amounts
  ep.ind <- seq(from=1, by=2, length=num.episodic)
  ep.amt <- seq(from=2, by=2, length=num.episodic)
  dy.amt <- seq(from=2*num.episodic+1, by=1, length=num.daily)

  ll.prob1 <- 0
  ll.prob2 <- 0
  ll.amt <- 0

  if(never.consumers.first.episodic) {

    num.consumption <- 0
    for(day.k in seq_len(num.recalls)) {

      if(recall.availability.i[day.k]) {

        num.consumption <- num.consumption + episodic.indicator.i[[day.k]][1]
      }
    }

    g.alpha <- sum(never.consumer.covariate.matrix.i*alpha1)

    ll.prob1 <- ll.prob1 + log(max(pnorm(g.alpha), sqrt(.Machine$double.eps)))
    if(num.consumption == 0) {

      ll.prob2 <- ll.prob2 + log(max(pnorm(-g.alpha), sqrt(.Machine$double.eps)))
    }
  }

  for(day.k in seq_len(num.recalls)) {

    if(recall.availability.i[day.k]) {

      xbeta.u <- xbeta.i[[day.k]] + u.matrix.i
      w.minus.xbeta.u <- w.matrix.i[[day.k]] - xbeta.u

      observed <- logical(num.variables)
      observed[ep.ind] <- FALSE
      observed[ep.amt] <- as.logical(episodic.indicator.i[[day.k]])
      observed[dy.amt] <- TRUE
      num.observed <- sum(observed)

      #Log-likelihood for probability part of model
      if(num.episodic > 0) {

        #Split multivariate normal distribution of W-XBeta-U into probability and amount parts using the Schur complement, integrating out latent W variables
        #Unobserved amounts are simply dropped because they will integrate to 1
        xbeta.u.prob <- xbeta.u[ep.ind]
        sigma.e.prob <- sigma.e[ep.ind, ep.ind, drop=FALSE]
        if(num.observed > 0) {

          xbeta.u.prob <- xbeta.u.prob + w.minus.xbeta.u[observed] %*% solve(sigma.e[observed, observed]) %*% sigma.e[observed, ep.ind]
          sigma.e.prob <- sigma.e.prob - sigma.e[ep.ind, observed] %*% solve(sigma.e[observed, observed]) %*% sigma.e[observed, ep.ind]
        }

        prob1 <- mvpnorm(x=ifelse(episodic.indicator.i[[day.k]], xbeta.u.prob, -xbeta.u.prob), sigma=sigma.e.prob)
        ll.prob1 <- ll.prob1 + log(max(prob1, sqrt(.Machine$double.eps)))

        if(never.consumers.first.episodic && num.consumption == 0) {

          prob2 <- prob1/pnorm(-xbeta.u.prob[1])
          ll.prob2 <- ll.prob2 + log(max(prob2, sqrt(.Machine$double.eps)))
        }
      }

      #Log-likelihood for amount part of model
      if(num.observed > 0) {

        w.minus.xbeta.u.amt <- w.minus.xbeta.u[observed]
        sigma.e.amt <- sigma.e[observed, observed, drop=FALSE]
        ll.amt <- ll.amt - 0.5*num.observed*log(2*pi) - 0.5*log(det(sigma.e.amt)) - 0.5*sum(w.minus.xbeta.u.amt*(w.minus.xbeta.u.amt %*% solve(sigma.e.amt)))
      }
    }
  }

  if(never.consumers.first.episodic && num.consumption == 0) {

    #Log-Sum-Exp trick to add consumer and never-consumer probabilities for potential never-consumers
    ll.prob.min <- min(ll.prob1, ll.prob2)
    ll.prob1 <- ll.prob1 - ll.prob.min
    ll.prob2 <- ll.prob2 - ll.prob.min

    ll.prob <- log(exp(ll.prob1) + exp(ll.prob2)) + ll.prob.min
  } else {

    ll.prob <- ll.prob1
  }

  ll <- ll.prob + ll.amt

  #Log-likelihood of random effects
  ll <- ll - 0.5*num.variables*log(2*pi) - 0.5*log(det(sigma.u)) - 0.5*sum(u.matrix.i*(u.matrix.i %*% solve(sigma.u)))

  return(ll)
}

#Calculates the marginal likelihood of multivariate NCI method MCMC model with random effects (U matrix) integrated out with Laplace approximation.
marginal_likelihood <- function(do.log.likelihood,
                                w.matrix,
                                episodic.indicator.matrices,
                                xbeta,
                                sigma.u,
                                sigma.e,
                                recall.availability,
                                subject.weighting,
                                num.subjects,
                                num.episodic,
                                num.daily,
                                num.recalls,
                                never.consumers.first.episodic,
                                never.consumer.covariate.matrix,
                                alpha1) {

  if(!do.log.likelihood) {

    return(NA)
  }

  num.variables <- 2*num.episodic + num.daily

  #Set precision levels for the finite difference derivatives and convergence tolerance
  if(num.episodic > 2) {

    #For more than two episodic variables, precision is bounded by the Monte Carlo algorithm used for the multivariate normal CDF
    delta <- rep((0.001)^(1/3), num.variables)
    epsilon <- sqrt(0.001)
  } else {

    #Use default machine precision if Monte Carlo algorithm is not used
    delta <- rep((.Machine$double.eps)^(1/3), num.variables)
    epsilon <- sqrt(.Machine$double.eps)
  }

  #Find joint log-likelihood at posterior mode of U matrix
  ll <- numeric(num.subjects)
  log.det.hessian <- numeric(num.subjects)
  for(i in seq_len(num.subjects)) {

    ll.opt <- optim(par=rep(0, num.variables),
                    fn=joint_likelihood,
                    w.matrix.i=lapply(w.matrix, function(w.k) w.k[i,]),
                    episodic.indicator.i=lapply(episodic.indicator.matrices, function(q.k) q.k[i,]),
                    xbeta.i=lapply(xbeta, function(xbeta.k) xbeta.k[i,]),
                    sigma.u=sigma.u,
                    sigma.e=sigma.e,
                    recall.availability.i=recall.availability[i,],
                    num.episodic=num.episodic,
                    num.daily=num.daily,
                    num.recalls=num.recalls,
                    never.consumers.first.episodic=never.consumers.first.episodic,
                    never.consumer.covariate.matrix.i=never.consumer.covariate.matrix[i,],
                    alpha1=alpha1,
                    method="BFGS",
                    control=list(fnscale=-1, ndeps=delta, reltol=epsilon),
                    hessian=TRUE)
    ll[i] <- ll.opt$value
    log.det.hessian[i] <- log(det(-ll.opt$hessian))
  }

  #Estimate marginal log-likelihood using Laplace approximation
  ll <- ll + 0.5*num.variables*log(2*pi) - 0.5*log.det.hessian

  #Return weighted sum of marginal-likelihood
  ll.total <- sum(ll*subject.weighting)
  return(ll.total)
}
