#Defines indicator and amount lists.
process_variable_lists <- function(pre.mcmc.data,
                                   episodic.variables,
                                   episodic.indicators,
                                   episodic.amounts,
                                   daily.variables,
                                   daily.amounts,
                                   never.consumer.variable,
                                   never.consumer.indicator,
                                   never.consumer.amount) {

  is.nci.multivar <- is(pre.mcmc.data, "nci.multivar.preprocessor")

  mcmc.input <- pre.mcmc.data$mcmc.input
  backtran.data <- pre.mcmc.data$backtransformation

  #if the input data is from the NCI multivar preprocessor, fill in indicators and amounts using appropriate prefixes
  if(is.nci.multivar) {

    if(length(episodic.indicators) == 0 && length(episodic.variables) > 0) {

      episodic.indicators <- paste0("ind.", episodic.variables)
    }

    if(length(episodic.amounts) == 0 && length(episodic.variables) > 0) {

      episodic.amounts <- paste0("amt.", episodic.variables)
    }

    if(length(daily.amounts) == 0 && length(daily.variables) > 0) {

      daily.amounts <- paste0("amt.", daily.variables)
    }

    if(length(never.consumer.indicator) == 0 && length(never.consumer.variable) > 0) {

      never.consumer.indicator <- paste0("ind.", never.consumer.variable)
    }

    if(length(never.consumer.amount) == 0 && length(never.consumer.variable) > 0) {

      never.consumer.amount <- paste0("amt.", never.consumer.variable)
    }
  }

  #if corresponding raw variables are not provided, then amount variables to backtransformation data
  if(length(episodic.amounts) > 0 && length(episodic.variables) == 0) {

    episodic.variables <- match_backtran(episodic.amounts, backtran.data$variable)
  }

  if(length(daily.amounts) > 0 && length(daily.variables) == 0) {

    daily.variables <- match_backtran(daily.amounts, backtran.data$variable)
  }

  if(length(never.consumer.amount) > 0 && length(never.consumer.variable) == 0) {

    never.consumer.variable <- match_backtran(never.consumer.amount, backtran.data$variable)
  }

  #place never-consumer variable as the first episodic variable (required by the MCMC algorithm)
  if(!is.null(never.consumer.variable)) {

    never.consumer.episodic.match <- (episodic.variables %in% never.consumer.variable)

    episodic.variables <- c(never.consumer.variable, episodic.variables[!never.consumer.episodic.match])
    episodic.indicators <- c(never.consumer.indicator, episodic.indicators[!never.consumer.episodic.match])
    episodic.amounts <- c(never.consumer.amount, episodic.amounts[!never.consumer.episodic.match])
  }

  #match variable lists to dataset
  episodic.variables <- match_variables(episodic.variables, backtran.data$variable)
  daily.variables <- match_variables(daily.variables, backtran.data$variable)

  episodic.indicators <- match_variables(episodic.indicators, mcmc.input)
  episodic.amounts <- match_variables(episodic.amounts, mcmc.input)
  daily.amounts <- match_variables(daily.amounts, mcmc.input)

  #Output processed variable lists
  variable.list <- list(episodic.variables=episodic.variables,
                        episodic.indicators=episodic.indicators,
                        episodic.amounts=episodic.amounts,
                        daily.variables=daily.variables,
                        daily.amounts=daily.amounts,
                        never.consumers.first.episodic=!is.null(never.consumer.variable))
  return(variable.list)
}

#Defines covariate lists for all indicators and amounts.
process_covariate_lists <- function(pre.mcmc.data,
                                    default.covariates,
                                    episodic.indicator.covariates,
                                    episodic.amount.covariates,
                                    daily.amount.covariates,
                                    individual.covariates,
                                    default.intercept,
                                    episodic.indicator.intercept,
                                    episodic.amount.intercept,
                                    daily.amount.intercept,
                                    individual.intercept,
                                    episodic.indicators,
                                    episodic.amounts,
                                    daily.amounts,
                                    never.consumers.first.episodic,
                                    never.consumer.covariates,
                                    never.consumer.intercept) {

  all.variables <- c(c(rbind(episodic.indicators, episodic.amounts)), daily.amounts)
  num.variables <- length(all.variables)

  covariate.list <- vector(mode="list", length=2)
  names(covariate.list) <- c("variables", "never.consumer")

  covariate.list$variables <- vector(mode="list", length=2)
  names(covariate.list$variables) <- c("covariates", "intercepts")

  covariate.list$variables$covariates <- vector(mode="list", length=num.variables)
  names(covariate.list$variables$covariates) <- all.variables
  covariate.list$variables$intercepts <- vector(mode="list", length=num.variables)
  names(covariate.list$variables$intercepts) <- all.variables

  for(var.j in seq_len(num.variables)) {

    variable <- all.variables[var.j]

    individual.covariate.name <- match_variables(variable, names(individual.covariates))
    if(!is.na(individual.covariate.name)) {

      covariates <- individual.covariates[[individual.covariate.name]]
    } else if((variable %in% episodic.indicators) && !is.null(episodic.indicator.covariates)) {

      covariates <- episodic.indicator.covariates
    } else if((variable %in% episodic.amounts) && !is.null(episodic.amount.covariates)) {

      covariates <- episodic.amount.covariates
    } else if((variable %in% daily.amounts) && !is.null(daily.amount.covariates)) {

      covariates <- daily.amount.covariates
    } else {

      covariates <- default.covariates
    }
    covariates <- match_variables(covariates, pre.mcmc.data$mcmc.input)

    individual.intercept.name <- match_variables(variable, names(individual.intercept))
    if(!is.na(individual.intercept.name)) {

      intercept <- individual.intercept[[individual.intercept.name]]
    } else if((variable %in% episodic.indicators) && !is.null(episodic.indicator.intercept)) {

      intercept <- episodic.indicator.intercept
    } else if((variable %in% episodic.amounts) && !is.null(episodic.amount.intercept)) {

      intercept <- episodic.amount.intercept
    } else if((variable %in% daily.amounts) && !is.null(daily.amount.intercept)) {

      intercept <- daily.amount.intercept
    } else {

      intercept <- default.intercept
    }

    covariate.list$variables$covariates[[var.j]] <- covariates
    covariate.list$variables$intercepts[[var.j]] <- intercept
  }

  if(never.consumers.first.episodic) {

    never.consumer.covariates <- match_variables(never.consumer.covariates, pre.mcmc.data$mcmc.input)

    covariate.list$never.consumer <- list(covariates=never.consumer.covariates,
                                          intercept=never.consumer.intercept)
  }

  return(covariate.list)
}

#Initializes variables, covariate matrices, and parameter priors for the MCMC loop.
initialize_mcmc <- function(pre.mcmc.data,
                            id,
                            repeat.obs,
                            weight,
                            variable.list,
                            covariate.list,
                            mcmc.seed,
                            sigma.u.prior) {

  #1. Set seed for MCMC
  if(is.null(mcmc.seed)) {

    mcmc.seed <- as.integer(runif(1, min=-0.5, max=0.5)*2*10^7)
  }
  set.seed(mcmc.seed)

  #2. Sort input data by subject and recall
  pre.mcmc.data$mcmc.input <- as.data.frame(pre.mcmc.data$mcmc.input)

  id <- match_variables(id, pre.mcmc.data$mcmc.input)
  repeat.obs <- match_variables(repeat.obs, pre.mcmc.data$mcmc.input)

  subject.recall.order <- order(pre.mcmc.data$mcmc.input[,id,drop=TRUE], pre.mcmc.data$mcmc.input[,repeat.obs,drop=TRUE])
  input.data <- pre.mcmc.data$mcmc.input[subject.recall.order,]

  #3. Define subject-level data
  subject.data <- initialize_subject_data(input.data=input.data,
                                          id=id,
                                          repeat.obs=repeat.obs,
                                          weight=weight,
                                          episodic.variables=variable.list$episodic.variables,
                                          daily.variables=variable.list$daily.variables)

  #4. Define variable matrices
  variable.matrices <- initialize_variable_matrices(input.data=input.data,
                                                    subjects=subject.data$subjects,
                                                    recalls=subject.data$recalls,
                                                    recall.availability=subject.data$recall.availability,
                                                    episodic.indicators=variable.list$episodic.indicator,
                                                    episodic.amounts=variable.list$episodic.amount,
                                                    daily.amounts=variable.list$daily.amount,
                                                    num.subjects=subject.data$num.subjects,
                                                    num.episodic=subject.data$num.episodic,
                                                    num.daily=subject.data$num.daily,
                                                    num.recalls=subject.data$num.recalls)

  #5. Define covariate matrices
  covariate.matrices <- initialize_covariate_matrices(input.data=input.data,
                                                      subjects=subject.data$subjects,
                                                      recalls=subject.data$recalls,
                                                      recall.availability=subject.data$recall.availability,
                                                      subject.weighting=subject.data$weighting,
                                                      covariates=covariate.list$variables$covariates,
                                                      intercepts=covariate.list$variables$intercepts,
                                                      never.consumers.first.episodic=variable.list$never.consumers.first.episodic,
                                                      never.consumer.covariates=covariate.list$never.consumer$covariates,
                                                      never.consumer.intercept=covariate.list$never.consumer$intercept,
                                                      num.subjects=subject.data$num.subjects,
                                                      num.episodic=subject.data$num.episodic,
                                                      num.daily=subject.data$num.daily,
                                                      num.recalls=subject.data$num.recalls)

  #6. Initialize MCMC priors
  mcmc.priors <- initialize_priors(episodic.indicator.matrices=variable.matrices$episodic.indicators,
                                   episodic.amount.matrices=variable.matrices$episodic.amounts,
                                   daily.amount.matrices=variable.matrices$daily.amounts,
                                   recall.covariate.matrices=covariate.matrices$recall,
                                   sigma.u.prior=sigma.u.prior,
                                   num.subjects=subject.data$num.subjects,
                                   num.episodic=subject.data$num.episodic,
                                   num.daily=subject.data$num.daily,
                                   num.recalls=subject.data$num.recalls,
                                   recall.availability=subject.data$recall.availability,
                                   never.consumers.first.episodic=variable.list$never.consumers.first.episodic,
                                   never.consumer.covariate.matrix=covariate.matrices$never.consumer)


  mcmc.parameters <- list(subject.data=subject.data,
                          covariate.matrices=covariate.matrices,
                          variable.matrices=variable.matrices,
                          priors=mcmc.priors,
                          mcmc.seed=mcmc.seed)
  return(mcmc.parameters)
}

#Splits input dataset by repeated observations and extracts subject, observation, and weight data.
initialize_subject_data <- function(input.data,
                                    id,
                                    repeat.obs,
                                    weight,
                                    episodic.variables,
                                    daily.variables) {

  #extract subjects and recalls from input data
  subjects <- input.data[,id, drop=TRUE]
  unique.subjects <- unique(subjects)
  num.subjects <- length(unique.subjects)

  recalls.raw <- input.data[,repeat.obs, drop=TRUE]
  unique.recalls <- unique(recalls.raw)
  recalls <- match(recalls.raw, unique.recalls)
  num.recalls <- length(unique.recalls)

  #recall availability for each subject
  recall.availability <- matrix(nrow=num.subjects, ncol=num.recalls)
  for(day.k in seq_len(num.recalls)) {

    recall.availability[,day.k] <- unique.subjects %in% subjects[recalls == day.k]
  }

  #extract subject weighting from first instance of each subject
  if(!is.null(weight)) {

    weight <- match_variables(weight, input.data)
    subject.weighting <- input.data[!duplicated(subjects), weight, drop=TRUE]
  } else {

    subject.weighting <- rep(1, length(unique.subjects))
  }
  subject.weighting <- subject.weighting/mean(subject.weighting)

  #number of episodic and daily variables
  num.episodic <- length(episodic.variables)
  num.daily <- length(daily.variables)

  #output subjects, recalls, and weights
  subject.level.data <- list(subjects=subjects,
                             recalls=recalls,
                             recall.availability=recall.availability,
                             weighting=subject.weighting,
                             num.subjects=num.subjects,
                             num.episodic=num.episodic,
                             num.daily=num.daily,
                             num.recalls=num.recalls)

  return(subject.level.data)
}

#Initializes model variable matrices for the MCMC function.
initialize_variable_matrices <- function(input.data,
                                         subjects,
                                         recalls,
                                         recall.availability,
                                         episodic.indicators,
                                         episodic.amounts,
                                         daily.amounts,
                                         num.subjects,
                                         num.episodic,
                                         num.daily,
                                         num.recalls) {

  episodic.indicator.data <- vector(mode="list", length=num.recalls)
  episodic.amount.data <- vector(mode="list", length=num.recalls)
  daily.amount.data <- vector(mode="list", length=num.recalls)
  for(day.k in seq_len(num.recalls)) {

    subjects.in.recall <- recall.availability[,day.k]
    obs.in.recall <- (recalls == day.k)

    #episodic indicator data
    episodic.indicator.data[[day.k]] <- matrix(0, nrow=num.subjects, ncol=length(episodic.indicators))
    episodic.indicator.data[[day.k]][subjects.in.recall,] <- as.matrix(input.data[obs.in.recall, episodic.indicators])
    colnames(episodic.indicator.data[[day.k]]) <- episodic.indicators

    #episodic amount data
    episodic.amount.data[[day.k]] <- matrix(0, nrow=num.subjects, ncol=length(episodic.amounts))
    episodic.amount.data[[day.k]][subjects.in.recall,] <- as.matrix(input.data[obs.in.recall, episodic.amounts])
    episodic.amount.data[[day.k]] <- ifelse(episodic.indicator.data[[day.k]], episodic.amount.data[[day.k]], 0)
    colnames(episodic.amount.data[[day.k]]) <- episodic.amounts

    #daily amount data
    daily.amount.data[[day.k]] <- matrix(0, nrow=num.subjects, ncol=length(daily.amounts))
    daily.amount.data[[day.k]][subjects.in.recall,] <- as.matrix(input.data[obs.in.recall, daily.amounts])
    colnames(daily.amount.data[[day.k]]) <- daily.amounts
  }

  #output response variables
  variable.matrices <- list(episodic.indicators=episodic.indicator.data,
                            episodic.amounts=episodic.amount.data,
                            daily.amounts=daily.amount.data)
  return(variable.matrices)
}

#Initializes covariate matrices for the MCMC function.
initialize_covariate_matrices <- function(input.data,
                                          subjects,
                                          recalls,
                                          recall.availability,
                                          subject.weighting,
                                          covariates,
                                          intercepts,
                                          never.consumers.first.episodic,
                                          never.consumer.covariates,
                                          never.consumer.intercept,
                                          num.subjects,
                                          num.episodic,
                                          num.daily,
                                          num.recalls) {

  num.variables <- 2*num.episodic + num.daily

  #create recall covariate matrices
  recall.covariate.matrices <- vector(mode="list", length=num.variables)
  for(var.j in seq_len(num.variables)) {

    recall.covariate.matrices[[var.j]] <- vector(mode="list", length=num.recalls)

    for(day.k in seq_len(num.recalls)) {

      subjects.in.recall <- recall.availability[,day.k]
      obs.in.recall <- (recalls == day.k)

      #covariates
      num.covariates <- length(covariates[[var.j]])
      covariate.matrix <- matrix(0, nrow=num.subjects, ncol=num.covariates)
      covariate.matrix[subjects.in.recall,] <- as.matrix(input.data[obs.in.recall, covariates[[var.j]]])
      colnames(covariate.matrix) <- covariates[[var.j]]

      #intercept
      if(intercepts[[var.j]]) {

        covariate.matrix <- cbind(intercept=as.numeric(subjects.in.recall), covariate.matrix)
      }

      recall.covariate.matrices[[var.j]][[day.k]] <- covariate.matrix
    }
  }
  names(recall.covariate.matrices) <- names(covariates)

  #calculate weighted covariate matrices
  weighted.recall.covariate.matrices <- vector(mode="list", length=num.variables)
  weighted.recall.covariate.squared.sums <- vector(mode="list", length=num.variables)
  for(var.j in seq_len(num.variables)) {

    num.covariates <- ncol(recall.covariate.matrices[[var.j]][[1]])

    #calculating weighted availability covariate matrices and sums for the current variable
    weighted.recall.covariate.matrices[[var.j]] <- vector(mode="list", length=num.recalls)
    weighted.recall.covariate.squared.sums[[var.j]] <- matrix(0, nrow=num.covariates, ncol=num.covariates)
    for(day.k in seq_len(num.recalls)) {

      weighted.recall.covariate.matrices[[var.j]][[day.k]] <- t(subject.weighting*recall.availability[,day.k]*recall.covariate.matrices[[var.j]][[day.k]])
      weighted.recall.covariate.squared.sums[[var.j]] <- weighted.recall.covariate.squared.sums[[var.j]] + weighted.recall.covariate.matrices[[var.j]][[day.k]] %*% recall.covariate.matrices[[var.j]][[day.k]]
    }
  }

  #create never-consumer covariate matrix
  if(never.consumers.first.episodic) {

    #extract never-consumer covariates from first instance of each subject
    never.consumer.covariate.matrix <- as.matrix(input.data[!duplicated(subjects), never.consumer.covariates])
    colnames(never.consumer.covariate.matrix) <- never.consumer.covariates

    if(never.consumer.intercept) {

      never.consumer.covariate.matrix <- cbind(intercept=1, never.consumer.covariate.matrix)
    }
  } else {

    never.consumer.covariate.matrix <- NULL
  }

  #output covariate data
  covariate.matrices <- list(recall=recall.covariate.matrices,
                             weighted.recall=weighted.recall.covariate.matrices,
                             weighted.covariate.squared.sums=weighted.recall.covariate.squared.sums,
                             never.consumer=never.consumer.covariate.matrix)

  return(covariate.matrices)
}


#Initializes MCMC priors and starting values - beta, sigma-e, r, theta, and sigma-u, W, U, and never-consumer parmaeters (alpha1 and consumer probabilities).
#Priors and starting values are defined in Appendices A.2 and A.3 of Zhang, et al. (2011) and Section 2.3.3 of Bhadra, et al. (2020) for never-consumers.
initialize_priors <- function(episodic.indicator.matrices,
                              episodic.amount.matrices,
                              daily.amount.matrices,
                              recall.covariate.matrices,
                              sigma.u.prior,
                              num.subjects,
                              num.episodic,
                              num.daily,
                              num.recalls,
                              recall.availability,
                              never.consumers.first.episodic,
                              never.consumer.covariate.matrix) {

  variables <- names(recall.covariate.matrices)
  num.variables <- 2*num.episodic + num.daily

  #initialize beta prior
  beta.mean.prior <- vector(mode="list", length=num.variables)
  beta.covariance.prior <- vector(mode="list", length=num.variables)
  for(var.j in seq_len(num.variables)) {

    #covariates for current variable
    covariates <- colnames(recall.covariate.matrices[[var.j]][[1]])
    num.covariates <- length(covariates)

    #beta mean prior (number of covariates x 1)
    beta.mean.prior[[var.j]] <- numeric(num.covariates)
    names(beta.mean.prior[[var.j]]) <- covariates

    #beta covariance prior (number of covariates x number of covariates)
    beta.covariance.prior[[var.j]] <- 100*diag(num.covariates)
    dimnames(beta.covariance.prior[[var.j]]) <- list(covariates, covariates)
  }
  names(beta.mean.prior) <- variables
  names(beta.covariance.prior) <- variables
  beta.prior <- list(mean=beta.mean.prior, covariance=beta.covariance.prior)

  #initialize r matrix prior (number of episodically consumed foods - 1 x 1) and theta matrix prior ((number of episodically consumed foods - 1)^2 x 1)
  if(num.episodic >= 2) {
    r.matrix.prior <- numeric(num.episodic - 1)
    theta.matrix.prior <- numeric((num.episodic - 1)^2)
  } else {
    r.matrix.prior <- NULL
    theta.matrix.prior <- NULL
  }

  #Initialize V matrix prior (number of episodic and daily variables x number of episodic and daily variables)
  v.matrix.prior <- diag(num.variables)
  dimnames(v.matrix.prior) <- list(variables, variables)

  #Initialize sigma-e prior (number of episodic and daily variables x number of episodic and daily variables)
  sigma.e.prior <- v.matrix.prior %*% t(v.matrix.prior)

  #Initialize sigma-u prior if it is not already given (number of episodic and daily variables x number of episodic and daily variables)
  if(is.null(sigma.u.prior)) {
    sigma.u.prior <- matrix(0.5, nrow=num.variables, ncol=num.variables) + 0.5*diag(num.variables)
  }
  dimnames(sigma.u.prior) <- list(variables, variables)

  #Initialize U matrix prior (number of subjects x number of episodic and daily variables)
  u.prior.normals <- matrix(rnorm(num.subjects*num.variables), nrow=num.subjects, ncol=num.variables)
  sigma.u.eigen <- eigen(sigma.u.prior)
  sqrt.sigma.u <- sigma.u.eigen$vectors %*% diag(sqrt(sigma.u.eigen$values)) %*% t(sigma.u.eigen$vectors)
  u.matrix.prior <- u.prior.normals %*% sqrt.sigma.u
  colnames(u.matrix.prior) <- variables


  #Initialize XBeta prior (number of subjects x number of episodic and daily variables for each recall)
  xbeta.prior <- vector(mode="list", length=num.recalls)
  for(day.k in seq_len(num.recalls)) {

    xbeta.prior[[day.k]] <- matrix(nrow=num.subjects, ncol=num.variables)
    for(var.j in seq_len(num.variables)) {

      xbeta.prior[[day.k]][,var.j] <- recall.covariate.matrices[[var.j]][[day.k]] %*% beta.mean.prior[[var.j]]
    }
    colnames(xbeta.prior[[day.k]]) <- variables
  }

  #Initialize XBeta_U prior (number of subjects x number of episodic and daily variables for each recall)
  xbeta.u.prior <- vector(mode="list", length=num.recalls)
  for(day.k in seq_len(num.recalls)) {

    xbeta.u.prior[[day.k]] <- xbeta.prior[[day.k]] + u.matrix.prior
  }

  #Initialize W matrix prior (number of subjects x number of episodic and daily variables for each recall)
  w.matrix.prior <- vector(mode="list", length=num.recalls)
  for(day.k in seq_len(num.recalls)) {

    w.matrix.prior[[day.k]] <- matrix(nrow=num.subjects, ncol=num.variables)
    for(var.j in seq_len(num.variables)) {

      if(var.j <= 2*num.episodic && var.j %% 2 == 1) {

        food.j <- (var.j + 1) %/% 2
        w.matrix.prior[[day.k]][,var.j] <- (2*episodic.indicator.matrices[[day.k]][,food.j] - 1)*abs(xbeta.u.prior[[day.k]][,var.j] + rnorm(num.subjects))*recall.availability[,day.k]
      } else if(var.j <= 2*num.episodic && var.j %% 2 == 0) {

        food.j <- (var.j + 1) %/% 2
        w.matrix.prior[[day.k]][,var.j] <- episodic.amount.matrices[[day.k]][,food.j]
      } else {

        nutr.j <- var.j - 2*num.episodic
        w.matrix.prior[[day.k]][,var.j] <- daily.amount.matrices[[day.k]][,nutr.j]
      }
    }
    colnames(w.matrix.prior[[day.k]]) <- variables
  }
  w.matrix.prior <- update_w_matrix(w.matrix=w.matrix.prior,
                                    xbeta.u=xbeta.u.prior,
                                    sigma.e=sigma.e.prior,
                                    recall.availability=recall.availability,
                                    episodic.indicator.matrices=episodic.indicator.matrices,
                                    num.subjects=num.subjects,
                                    num.episodic=num.episodic,
                                    num.daily=num.daily,
                                    num.recalls=num.recalls)

  #never-consumers priors
  if(never.consumers.first.episodic) {

    never.consumer.covariates <- colnames(never.consumer.covariate.matrix)
    num.never.consumer.covariates <- length(never.consumer.covariates)

    #Initialize alpha1 mean (number of never-consumer covariates x 1) and covariance (number of never-consumer covariates x number of never-consumer covariates) priors
    alpha1.mean.prior <- numeric(num.never.consumer.covariates)
    names(alpha1.mean.prior) <- never.consumer.covariates

    alpha1.covariance.prior <- diag(num.never.consumer.covariates)
    dimnames(alpha1.covariance.prior) <- list(never.consumer.covariates, never.consumer.covariates)

    #Initialize consumer probabilities prior (number of subjects x 1)
    g.alpha.prior <- as.vector(never.consumer.covariate.matrix %*% alpha1.mean.prior)
    consumer.probabilities.prior <- pnorm(g.alpha.prior)
  } else {

    alpha1.mean.prior <- NULL
    alpha1.covariance.prior <- NULL
    consumer.probabilities.prior <- NULL
  }
  alpha1.prior <- list(mean=alpha1.mean.prior, covariance=alpha1.covariance.prior)

  #output as list
  list(alpha1.prior=alpha1.prior,
       consumer.probabilities.prior=consumer.probabilities.prior,
       beta.prior=beta.prior,
       sigma.e.prior=sigma.e.prior,
       r.matrix.prior=r.matrix.prior,
       theta.matrix.prior=theta.matrix.prior,
       v.matrix.prior=v.matrix.prior,
       sigma.u.prior=sigma.u.prior,
       u.matrix.prior=u.matrix.prior,
       xbeta.prior=xbeta.prior,
       xbeta.u.prior=xbeta.u.prior,
       w.matrix.prior=w.matrix.prior)
}
