#' Significance Testing for MCMC Covariates
#'
#' @description Tests the significance of covariates in an MCMC model using
#'   replication techniques.
#'
#' @details The variance/covariance matrix of the beta parameters in the MCMC
#'   model is found by fitting MCMC model replicates with bootstrap or balanced
#'   repeated replication (BRR). The original weighting variable for the dataset
#'   is given in `base.weight` and the replicate weighting variables are given
#'   in `rep.weights`. For survey datasets that BRR is applicable to, BRR
#'   weights can be generated by [brr_weights()]. When using BRR, the value of
#'   `fay.factor` must match the Fay factor used to generate the weights.
#'
#'   The covariates to test are specified in `test.covariates`. Elements of
#'   `test.covariates` that are not present in the model are ignored. The
#'   `test.covariates` can include covariates from the episodic indicator,
#'   episodic amount, and daily amount parts of the model. This function does
#'   not test never-consumer covariates.
#'
#'   The `df` parameter is the number of degrees of freedom to use for the
#'   Hotelling's t-squared test. For survey datasets, this is the total number
#'   of PSUs across all strata minus the number of strata.
#'
#' @inheritParams nci_multivar_mcmc
#' @param base.weight Variable with the original weighting for each subject.
#' @param initial.mcmc.seed Numeric starting seed for the random number
#'   generator. If specified, the seed will be incremented for each MCMC
#'   replicate. If `NULL`, uses a randomly generated integer from -10^7 to 10^7,
#'   exclusive for each replicate. (default = `NULL`)
#' @param rep.weights Vector of variables with the replicate weights
#'   for each subject.
#' @param brr Flag specifying whether replicate weights are for balanced
#'   repeated replication (BRR). (default = `FALSE`)
#' @param fay.factor Numeric between 0 and 1 specifying the Fay factor used to
#'   generate the BRR weights used in the function. Applies only when `brr` is
#'   `TRUE`. (default = `1`)
#' @param df Integer specifying the number of degrees of freedom for Hotelling's
#'   T^2 test.
#' @param test.covariates Vector of covariates to be tested.
#'
#' @returns A single joint p-value for all of the covariates in
#'   `test.covariates`.
#'
#' @export
#'
#' @examples
#' #subset NHANES data
#' nhanes.subset <- nhcvd[nhcvd$SDMVSTRA %in% c(48, 60, 72),]
#'
#' #generate BRR weights for the dataset
#' brr.data <- brr_weights(input.data=nhanes.subset,
#'                         id="SEQN",
#'                         strata="SDMVSTRA",
#'                         psu="SDMVPSU",
#'                         cell="PSCELL",
#'                         weight="WTDRD1",
#'                         fay.factor=0.7)
#'
#' boxcox.lambda.data <- boxcox_survey(input.data=brr.data,
#'                                     row.subset=(brr.data$DAY == 1),
#'                                     variable="TSODI",
#'                                     id="SEQN",
#'                                     repeat.obs="DAY",
#'                                     weight="RepWt_0",
#'                                     covariates="RIDAGEYR")
#'
#' minimum.amount.data <- calculate_minimum_amount(input.data=brr.data,
#'                                                 row.subset=(brr.data$DAY == 1),
#'                                                 daily.variables="TSODI")
#'
#' pre.mcmc.data <- nci_multivar_preprocessor(input.data=brr.data,
#'                                            daily.variables="TSODI",
#'                                            continuous.covariates="RIDAGEYR",
#'                                            boxcox.lambda.data=boxcox.lambda.data,
#'                                            minimum.amount.data=minimum.amount.data)
#'
#' #testing whether age is significant in the model for sodium
#' df <- length(unique(brr.data$SDMVSTRA))
#' pval.age <- covariate_test(pre.mcmc.data=pre.mcmc.data,
#'                            id="SEQN",
#'                            repeat.obs="DAY",
#'                            base.weight="RepWt_0",
#'                            daily.variables="TSODI",
#'                            default.covariates="std.RIDAGEYR",
#'                            num.mcmc.iterations=1000,
#'                            num.burn=500,
#'                            num.thin=1,
#'                            rep.weights=paste0("RepWt_", 1:4),
#'                            brr=TRUE,
#'                            fay.factor=0.7,
#'                            df=df,
#'                            test.covariates="std.RIDAGEYR")
#' pval.age
covariate_test <- function(pre.mcmc.data,
                           id,
                           repeat.obs,
                           base.weight=NULL,
                           episodic.variables=NULL,
                           episodic.indicators=NULL,
                           episodic.amounts=NULL,
                           daily.variables=NULL,
                           daily.amounts=NULL,
                           default.covariates=NULL,
                           episodic.indicator.covariates=NULL,
                           episodic.amount.covariates=NULL,
                           daily.amount.covariates=NULL,
                           individual.covariates=list(),
                           default.intercept=TRUE,
                           episodic.indicator.intercept=NULL,
                           episodic.amount.intercept=NULL,
                           daily.amount.intercept=NULL,
                           individual.intercept=list(),
                           never.consumer.variable=NULL,
                           never.consumer.indicator=NULL,
                           never.consumer.amount=NULL,
                           never.consumer.covariates=NULL,
                           never.consumer.intercept=TRUE,
                           initial.mcmc.seed=NULL,
                           num.mcmc.iterations=12000,
                           num.burn=2000,
                           num.thin=25,
                           sigma.u.prior=NULL,
                           sigma.u.constant=FALSE,
                           rep.weights,
                           brr=FALSE,
                           fay.factor=1,
                           df,
                           test.covariates) {

  mcmc.seed <- initial.mcmc.seed

  #run MCMC with base weights and all replicate weights and extract chosen beta parameters
  num.replicates <- length(rep.weights)
  replicate.list <- vector(mode="list", length=num.replicates)
  for(i in 0:num.replicates) {

    if(i == 0) {

      print("MCMC for Point Estimate")
      weight <- base.weight
    } else {

      print(paste0("MCMC for Replicate ", i))
      weight <- rep.weights[i]
    }

    #Run MCMC
    mcmc.output <- nci_multivar_mcmc(pre.mcmc.data=pre.mcmc.data,
                                     id=id,
                                     repeat.obs=repeat.obs,
                                     weight=weight,
                                     episodic.variables=episodic.variables,
                                     episodic.indicators=episodic.indicators,
                                     episodic.amounts=episodic.amounts,
                                     daily.variables=daily.variables,
                                     daily.amounts=daily.amounts,
                                     default.covariates=default.covariates,
                                     episodic.indicator.covariates=episodic.indicator.covariates,
                                     episodic.amount.covariates=episodic.amount.covariates,
                                     daily.amount.covariates=daily.amount.covariates,
                                     individual.covariates=individual.covariates,
                                     default.intercept=default.intercept,
                                     episodic.indicator.intercept=episodic.indicator.intercept,
                                     episodic.amount.intercept=episodic.amount.intercept,
                                     daily.amount.intercept=daily.amount.intercept,
                                     individual.intercept=individual.intercept,
                                     never.consumer.variable=never.consumer.variable,
                                     never.consumer.indicator=never.consumer.indicator,
                                     never.consumer.amount=never.consumer.amount,
                                     never.consumer.covariates=never.consumer.covariates,
                                     never.consumer.intercept=never.consumer.intercept,
                                     mcmc.seed=mcmc.seed,
                                     num.mcmc.iterations=num.mcmc.iterations,
                                     num.burn=num.burn,
                                     num.thin=num.thin,
                                     sigma.u.prior=sigma.u.prior,
                                     sigma.u.constant=sigma.u.constant)

    #Select beta parameters for covariates to test
    if(i == 0) {

      num.test <- length(test.covariates)
      num.variables <- 2*mcmc.output$num.episodic + mcmc.output$num.daily

      beta.selected <- character(0)
      for(var.j in seq_len(num.variables)) {

        variable <- names(mcmc.output$covariates)[var.j]
        covariates <- mcmc.output$covariates[[var.j]]

        beta.names <- paste0("beta", ".", variable, ".", covariates)
        beta.selected <- c(beta.selected, beta.names[covariates %in% test.covariates])
      }

      num.selected <- length(beta.selected)
    }

    #Extract selected beta parameter means
    mcmc.parameters <- extract_parameters(mcmc.output)
    mcmc.means <- mcmc.parameters$mean

    selected <- mcmc.means[beta.selected]

    if(i == 0) {

      point.estimate <- selected
    } else {

      replicate.list[[i]] <- selected
    }

    #increment seed if it is specified
    if(!is.null(mcmc.seed)) {

      mcmc.seed <- mcmc.seed + 1
    }
  }
  replicates <- do.call(cbind, replicate.list)

  #calculate variance-covariance matrix of beta estimates
  if(!brr) {

    fay.factor <- 1
  }

  beta.covariance <- ((replicates - point.estimate) %*% t(replicates - point.estimate))/(num.replicates*fay.factor^2)

  #Hotelling T^2 test
  t2 <- as.vector(t(point.estimate) %*% solve(beta.covariance) %*% point.estimate)

  N <- df
  p <- num.selected

  F.statistic <- t2*(N-p)/(p*(N-1))
  p.value <- 1 - pf(F.statistic, p, N-p)

  return(p.value)
}
