#Simulates random intercepts ('u' matrix) for each subject.
generate_u_matrix <- function(use.mcmc.u.matrices,
                              mcmc.u.matrix,
                              u.standard.deviation,
                              mcmc.u.index,
                              mcmc.distrib.match,
                              sim.u.index,
                              num.subjects,
                              num.episodic,
                              num.daily) {

  #total number of episodic and daily variables
  num.variables <- 2*num.episodic + num.daily

  if(use.mcmc.u.matrices) {

    u.matrix <- matrix(nrow=num.subjects, ncol=num.variables)

    #if using MCMC U matrices, use the U matrix for the current replicate as a base
    if(!is.null(mcmc.u.index)) {

      u.matrix[mcmc.u.index,] <- mcmc.u.matrix[mcmc.distrib.match,]
    }

    #simulate remaining observations if needed
    if(!is.null(sim.u.index)) {

      num.sim <- length(sim.u.index)
      normals <- matrix(rnorm(num.sim*num.variables), nrow=num.sim, ncol=num.variables)
      u.matrix[sim.u.index,] <- normals %*% u.standard.deviation
    }
  } else {

    normals <- matrix(rnorm(num.subjects*num.variables), nrow=num.subjects, ncol=num.variables)
    u.matrix <- normals %*% u.standard.deviation
  }

  return(u.matrix)
}


#For models with never-consumers allowed for the first episodic variable, identifies which subjects are never-consumers.
find_never_consumers <- function(never.consumers.first.episodic,
                                 consumer.probabilities,
                                 subject.record.match,
                                 num.subjects) {

  if(never.consumers.first.episodic) {

    never.consumer.selection <- runif(num.subjects)
    never.consumers <- (never.consumer.selection[subject.record.match] > consumer.probabilities)
  } else {

    never.consumers <- NULL
  }

  return(never.consumers)
}



#Calculates the probability of consumption for episodic variables.
calculate_consumption_probability <- function(xbeta.u,
                                              episodic.variables,
                                              num.records,
                                              num.episodic,
                                              never.consumers.first.episodic,
                                              never.consumers) {

  if(num.episodic == 0) {

    return(NULL)
  }

  #Calculate consumption probability of episodic variables
  consumption.probability <- matrix(nrow=num.records, ncol=num.episodic)

  for(i in seq_len(num.episodic)) {

    consumption.probability[,i] <- pnorm(xbeta.u[,2*i-1])
  }

  #if never-consumers are allowed, set consumption probabilities for never-consumers to zero for the first episodic food
  if(never.consumers.first.episodic) {

    consumption.probability[never.consumers,1] <- 0
  }

  colnames(consumption.probability) <- paste0("prob.", episodic.variables)

  return(consumption.probability)
}

#Calculates the consumption amount on the original scale for episodic and daily variables.
calculate_backtransformed_amount <- function(xbeta.u,
                                             sigma.e.mean,
                                             backtransformation.data,
                                             episodic.variables,
                                             daily.variables,
                                             num.records,
                                             num.episodic,
                                             num.daily,
                                             never.consumers.first.episodic,
                                             never.consumers) {

  variables <- c(episodic.variables, daily.variables)

  #calculate consumption amount on the original scale for all variables
  backtransformed.amount <- matrix(nrow=num.records, ncol=num.episodic+num.daily)
  for(i in seq_len(num.episodic+num.daily)) {

    backtran.index <- which(backtransformation.data$variable == variables[i])

    if(i <= num.episodic) {

      amount.index <- 2*i
    } else {

      amount.index <- num.episodic + i;
    }

    backtransformed.amount[,i] <- backtransform(xbeta.u.var=xbeta.u[,amount.index],
                                                sigma.e.var=sigma.e.mean[amount.index, amount.index],
                                                lambda=backtransformation.data$tran_lambda[backtran.index],
                                                center=backtransformation.data$tran_center[backtran.index],
                                                scale=backtransformation.data$tran_scale[backtran.index],
                                                minimum.amount=backtransformation.data$minamount[backtran.index],
                                                is.biomarker=backtransformation.data$biomarker[backtran.index])
  }

  #if never-consumers are allowed, set amounts for never-consumers to missing for the first episodic food
  if(never.consumers.first.episodic && num.episodic > 0) {

    backtransformed.amount[never.consumers,1] <- NA
  }

  colnames(backtransformed.amount) <- paste0("amount.", variables)

  return(backtransformed.amount)
}



#Calculates usual intakes from consumption probabilities and amounts.
calculate_usual_intake <- function(consumption.probability,
                                   backtransformed.amount,
                                   episodic.variables,
                                   daily.variables,
                                   num.records,
                                   num.episodic,
                                   num.daily,
                                   never.consumers.first.episodic,
                                   never.consumers) {

  #calculate usual intake for episodic and daily variables
  usual.intake <- matrix(nrow=num.records, ncol=num.episodic+num.daily)

  if(num.episodic > 0) {

    usual.intake[,1:num.episodic] <- consumption.probability*backtransformed.amount[,1:num.episodic]
  }

  if(num.daily > 0) {

    usual.intake[,(num.episodic+1):(num.episodic+num.daily)] <- backtransformed.amount[,(num.episodic+1):(num.episodic+num.daily)]
  }

  #if never-consumers are allowed, set usual intake for never-consumers to zero
  if(never.consumers.first.episodic && num.episodic > 0) {

    usual.intake[never.consumers,1] <- 0
  }

  variables <- c(episodic.variables, daily.variables)
  colnames(usual.intake) <- paste0("usual.intake.", variables)

  return(usual.intake)
}

#Adds dietary supplements to corresponding usual intakes.
calculate_supplemented_intake <- function(usual.intake,
                                          dietary.supplement.data,
                                          episodic.variables,
                                          daily.variables,
                                          num.records) {

  if(is.null(dietary.supplement.data)) {

    return(NULL)
  }

  variables <- c(episodic.variables, daily.variables)

  variables.to.supplement <- colnames(dietary.supplement.data)
  num.supplemented <- ncol(dietary.supplement.data)

  supplemented.intake <- matrix(nrow=num.records, ncol=num.supplemented)
  for(i in seq_len(num.supplemented)) {

    variable.index <- which(variables == variables.to.supplement[i])
    supplemented.intake[,i] <- usual.intake[,variable.index] + dietary.supplement.data[,i]
  }

  colnames(supplemented.intake) <- paste0("supplemented.intake.", variables.to.supplement)

  return(supplemented.intake)
}


#Calculates consumption amounts backtransformed to the original scale from a mixed model predictor of a Box-Cox transformed variable.
backtransform <- function(xbeta.u.var,
                          sigma.e.var,
                          lambda,
                          center,
                          scale,
                          minimum.amount,
                          is.biomarker) {

  if(is.biomarker) {

    boxcox.amounts <- center + scale*xbeta.u.var
    backtransformed.amounts <- inverse_box_cox(boxcox.amounts, lambda, minimum.amount)
  } else {

    if(lambda == 0) {

      boxcox.amounts <- center + scale*xbeta.u.var + (scale^2)*sigma.e.var/2
      backtransformed.amounts <- inverse_box_cox(boxcox.amounts, 0, minimum.amount)
    } else {

      #perform 9-point approximation for non-zero lambda
      x9pt <- c(-2.1, -1.3, -0.8, -0.5, 0, 0.5, 0.8, 1.3, 2.1)
      w9pt <- c(0.063345, 0.080255, 0.070458, 0.159698, 0.252489, 0.159698, 0.070458, 0.080255, 0.063345)
      w9pt <- w9pt/sum(w9pt)

      backtransformed.amounts <- numeric(length(xbeta.u.var))
      for(i in 1:9) {

        boxcox.amounts <- center + scale*(xbeta.u.var + x9pt[i]*sqrt(sigma.e.var))
        backtransformed.amounts <- backtransformed.amounts + w9pt[i]*inverse_box_cox(boxcox.amounts, lambda, minimum.amount)
      }
    }
  }

  return(backtransformed.amounts)
}


#Converts Box-Cox transformed values back to the original scale.
inverse_box_cox <- function(boxcox.values, lambda, minimum.amount) {

  if(lambda == 0) {

    inverse.boxcox <- exp(boxcox.values)
  } else{

    inverse.boxcox <- pmax(1 + lambda*boxcox.values, 0)^(1/lambda)
  }

  inverse.boxcox <- pmax(inverse.boxcox, minimum.amount)

  return(inverse.boxcox)
}
