#' Weighted quantiles
#'
#' @description Calculates weighted quantiles.
#'
#' @details Weighted quantiles are calculated using the empirical distribution
#'   function with weights, averaging at discontinuities. This is equivalent to
#'   weighted percentiles with PROC UNIVARIATE in SAS.
#'
#'
#' @param x A numeric vector.
#' @param w Numeric vector of weights the same length as `x`.
#' @param probs Numeric vector of quantiles to compute. Can range from 0 to 1.
#'
#' @returns A numeric vector of length `length(probs)` with the computed
#'   quantiles of `x`.
#'
#' @export
#'
#' @examples
#' #Weighted percentiles of reported energy intake
#' weighted_quantiles(x=nhcvd$TKCAL, w=nhcvd$WTDRD1, probs=c(0.05, 0.1, 0.25, 0.5, 0.75, 0.9, 0.95))
weighted_quantiles <- function(x, w, probs) {

  x.sorted <- x[order(x)]
  w.sorted <- w[order(x)]

  sum.weights <- sum(w.sorted)
  cumulative.weights <- cumsum(w.sorted)

  index <- findInterval(probs*sum.weights, cumulative.weights, all.inside=TRUE)

  fractional.part <- probs*sum.weights - cumulative.weights[index]

  below.interval <- (fractional.part < 0)
  above.interval <- (fractional.part > 0)
  equal.interval <- (fractional.part == 0)

  quantile.below.interval <- x.sorted[index]
  quantile.above.interval <- x.sorted[index+1]
  quantile.equal.interval <- (x.sorted[index] + x.sorted[index+1])/2

  weighted.quantile <- quantile.below.interval*below.interval +
                       quantile.above.interval*above.interval +
                       quantile.equal.interval*equal.interval
  names(weighted.quantile) <- paste0(100*probs, "%")

  return(weighted.quantile)
}
