#' MCMC Trace Plots
#'
#' @description Generate trace plots from an `nci.multivar.mcmc` object.
#'
#' @details Trace plots are a method of assessing the convergence of MCMC chain.
#'   The value of an MCMC parameter (such as a coefficient or a covariance
#'   matrix element) is plotted against the number of iterations to see how the
#'   value fluctuates over time. Plots of the parameter values and rolling
#'   means can be produced. A parameter value that moves around a fixed point with
#'   a uniform variance over time is said to have converged or "mixed well". For
#'   more information, please see the trace plots vignette.
#'
#'
#' @param multivar.mcmc.model An `nci.multivar.mcmc` object.
#' @param do.beta Flag to specify whether trace plots of fixed effect ('beta')
#'   coefficients should be produced. (default = `TRUE`)
#' @param do.sigma.e Flag to specify whether trace plots of residual error
#'   covariance matrix ('Sigma-e') elements should be produced. (default =
#'   `TRUE`)
#' @param do.sigma.u Flag to specify whether trace plots of random effect
#'   covariance matrix ('Sigma-u') elements should be produced. (default =
#'   `TRUE`)
#' @param do.sigma.overlay Flag to specify whether overlays of 'Sigma-e' and
#'   'Sigma-u' element trace plots should be produced. (default = `FALSE`)
#' @param do.alpha1 Flag to specify whether trace plots of never-consumer model
#'   ('alpha1') coefficients should be produced. No effect if no never-consumer
#'   variable was modeled. (default = `TRUE`)
#' @param do.consumer.probability Flag to specify whether trace plots of mean
#'   consumer probability should be produced. No effect if no never-consumer
#'   variable was modeled. (default = `TRUE`)
#' @param plot.type A character vector specifying what types of plots to
#'   generate. One or both of the following can be specified:
#'   * `"value"`: Plots of a parameter's value at every MCMC iteration. This is the standard type of trace plot. (default)
#'   * `"mean"`: Plots of the rolling mean of a parameter at thinned MCMC iterations after the burn-in period.
#'
#' @returns `NULL`. This function does not produce a return value, instead it
#'   outputs the trace plots that were created into PDF files in the current
#'   working directory.
#'
#' @export
#'
#' @examples
#' #subset NHANES data
#' nhanes.subset <- nhcvd[nhcvd$SDMVSTRA %in% c(48, 60, 72),]
#'
#' boxcox.lambda.data <- boxcox_survey(input.data=nhanes.subset,
#'                                     row.subset=(nhanes.subset$DAY == 1),
#'                                     variable="TSODI",
#'                                     id="SEQN",
#'                                     repeat.obs="DAY",
#'                                     weight="WTDRD1",
#'                                     covariates="RIDAGEYR")
#'
#' minimum.amount.data <- calculate_minimum_amount(input.data=nhanes.subset,
#'                                                 row.subset=(nhanes.subset$DAY == 1),
#'                                                 daily.variables="TSODI")
#'
#' pre.mcmc.data <- nci_multivar_preprocessor(input.data=nhanes.subset,
#'                                            daily.variables="TSODI",
#'                                            continuous.covariates="RIDAGEYR",
#'                                            boxcox.lambda.data=boxcox.lambda.data,
#'                                            minimum.amount.data=minimum.amount.data)
#'
#' mcmc.output <- nci_multivar_mcmc(pre.mcmc.data=pre.mcmc.data,
#'                                  id="SEQN",
#'                                  weight="WTDRD1",
#'                                  repeat.obs="DAY",
#'                                  daily.variables="TSODI",
#'                                  default.covariates="std.RIDAGEYR",
#'                                  num.mcmc.iterations=1000,
#'                                  num.burn=500,
#'                                  num.thin=1)
#'
#' #display trace plots
#' trace_plots(mcmc.output)
trace_plots <- function(multivar.mcmc.model,
                        do.beta=TRUE,
                        do.sigma.e=TRUE,
                        do.sigma.u=TRUE,
                        do.sigma.overlay=FALSE,
                        do.alpha1=TRUE,
                        do.consumer.probability=TRUE,
                        plot.type="value") {

  #types of plots to generate
  plot.type <- match.arg(plot.type, c("value", "mean"), several.ok=TRUE)

  #number of iterations and burn-in
  num.iterations <- multivar.mcmc.model$num.mcmc.iterations
  num.burn <- multivar.mcmc.model$num.burn
  num.thin <- multivar.mcmc.model$num.thin

  #get variables to make trace plots for
  variables <- c(rbind(multivar.mcmc.model$episodic.indicators, multivar.mcmc.model$episodic.amounts), multivar.mcmc.model$daily.amounts)

  #beta trace plots
  if(do.beta) {

    beta_trace_plots(beta=multivar.mcmc.model$beta,
                     variables=variables,
                     num.iterations=num.iterations,
                     num.burn=num.burn,
                     num.thin=num.thin,
                     plot.type=plot.type)
  }


  #Sigma-e trace plots
  if(do.sigma.e) {

    sigma_trace_plots(sigma=multivar.mcmc.model$sigma.e,
                      component="e",
                      variables=variables,
                      num.iterations=num.iterations,
                      num.burn=num.burn,
                      num.thin=num.thin,
                      plot.type=plot.type)
  }


  #Sigma-u trace plots
  if(do.sigma.u) {

    sigma_trace_plots(sigma=multivar.mcmc.model$sigma.u,
                      component="u",
                      variables=variables,
                      num.iterations=num.iterations,
                      num.burn=num.burn,
                      num.thin=num.thin,
                      plot.type=plot.type)
  }

  #sigma overlay plots
  if(do.sigma.overlay) {

    sigma_overlay_plots(sigma.e=multivar.mcmc.model$sigma.e,
                        sigma.u=multivar.mcmc.model$sigma.u,
                        variables=variables,
                        num.iterations=num.iterations,
                        num.burn=num.burn,
                        num.thin=num.thin,
                        plot.type=plot.type)
  }

  #Never-consumer trace plots
  if(multivar.mcmc.model$never.consumers.first.episodic) {

    #alpha1 trace plots
    if(do.alpha1) {

      alpha1_trace_plots(alpha1=multivar.mcmc.model$alpha1,
                         num.iterations=num.iterations,
                         num.burn=num.burn,
                         num.thin=num.thin,
                         plot.type=plot.type)
    }


    #consumer probability trace plots
    if(do.consumer.probability) {

      consumer_probability_trace_plots(consumer.probabilities=multivar.mcmc.model$consumer.probabilities,
                                       num.iterations=num.iterations,
                                       num.burn=num.burn,
                                       num.thin=num.thin,
                                       plot.type=plot.type)
    }
  }

  return(NULL)
}

#Produce trace plots of beta values and means.
beta_trace_plots <- function(beta,
                             variables,
                             num.iterations,
                             num.burn,
                             num.thin,
                             plot.type) {

  for(var.j in seq_along(variables)) {

    covariates <- rownames(beta[[var.j]])
    for(covar.j in seq_along(covariates)) {

      beta.value <- beta[[var.j]][covar.j,]

      if("value" %in% plot.type) {

        value.plot.title <- paste0("Beta Values, ", variables[var.j], ", ", covariates[covar.j])
        plot(x=seq_along(beta.value),
             y=beta.value,
             type="l",
             main=value.plot.title,
             xlab="Number of Iterations",
             ylab="Beta Value")
        abline(v=num.burn, col="red")
      }

      if("mean" %in% plot.type) {

        thinned.iterations <- seq(num.burn+1, num.iterations, num.thin)
        beta.thinned <- beta.value[thinned.iterations]
        beta.mean <- cumsum(beta.thinned)/seq_along(beta.thinned)

        mean.plot.title <- paste0("Beta Means, ", variables[var.j], ", ", covariates[covar.j])
        plot(x=thinned.iterations,
             y=beta.mean,
             type="l",
             main=mean.plot.title,
             xlab="Number of Iterations",
             ylab="Beta Mean")
      }
    }
  }

  return(NULL)
}

#Produces trace plots of Sigma-u and Sigma-e values and means.
sigma_trace_plots <- function(sigma,
                              component,
                              variables,
                              num.iterations,
                              num.burn,
                              num.thin,
                              plot.type) {

  for(row.num in seq_along(variables)) {

    for(col.num in seq_len(row.num)) {

      sigma.value <- sigma[row.num, col.num,]

      if("value" %in% plot.type) {

        value.plot.title <- paste0("Sigma-", component, " Values, ", variables[row.num], ", ", variables[col.num])
        plot(x=seq_along(sigma.value),
             y=sigma.value,
             type="l",
             main=value.plot.title,
             xlab="Number of Iterations",
             ylab=paste0("Sigma-", component, " Value"))
        abline(v=num.burn, col="red")
      }


      if("mean" %in% plot.type) {

        thinned.iterations <- seq(num.burn+1, num.iterations, num.thin)
        sigma.thinned <- sigma.value[thinned.iterations]
        sigma.mean <- cumsum(sigma.thinned)/seq_along(sigma.thinned)

        mean.plot.title <- paste0("Sigma-", component, " Mean, ", variables[row.num], ", ", variables[col.num])
        plot(x=thinned.iterations,
             y=sigma.mean,
             type="l",
             main=mean.plot.title,
             xlab="Number of Iterations",
             ylab=paste0("Sigma-", component, " Mean"))
      }
    }
  }

  return(NULL)
}

#Produces overlay of Sigma-u and Sigma-e trace plots
sigma_overlay_plots <- function(sigma.u,
                                sigma.e,
                                variables,
                                num.iterations,
                                num.burn,
                                num.thin,
                                plot.type) {

  for(row.num in seq_along(variables)) {

    for(col.num in seq_len(row.num)) {

      sigma.u.value <- sigma.u[row.num, col.num,]
      sigma.e.value <- sigma.e[row.num, col.num,]

      if("value" %in% plot.type) {

        overlay.value.title <- paste0("Overlay of Sigma-u and Sigma-e Values, ", variables[row.num], ", ", variables[col.num])
        max.value <- max(sigma.e.value, sigma.u.value)
        min.value <- min(sigma.e.value, sigma.u.value)
        value.range <- max.value - min.value
        upper.limit.value <- max.value + 0.05*value.range
        lower.limit.value <- min.value - 0.2*value.range

        plot(x=seq_along(sigma.e.value),
             y=sigma.e.value,
             type="l",
             main=overlay.value.title,
             xlab="Number of Iterations",
             ylab="Parameter Value",
             ylim=c(lower.limit.value, upper.limit.value))
        lines(x=seq_along(sigma.u.value),
              y=sigma.u.value,
              col="blue")
        abline(v=num.burn, col="red")
        legend(x="bottomright",
               legend=c("Sigma-e", "Sigma-u"),
               col=c("black", "blue"),
               lty=c(1,1),
               lwd=c(2,2),
               bty="n")
      }

      if("mean" %in% plot.type) {

        thinned.iterations <- seq(num.burn+1, num.iterations, num.thin)

        sigma.u.thinned <- sigma.u.value[thinned.iterations]
        sigma.e.thinned <- sigma.e.value[thinned.iterations]

        sigma.u.mean <- cumsum(sigma.u.thinned)/seq_along(sigma.u.thinned)
        sigma.e.mean <- cumsum(sigma.e.thinned)/seq_along(sigma.e.thinned)

        overlay.mean.title <- paste0("Overlay of Sigma-u and Sigma-e Means, ", variables[row.num], ", ", variables[col.num])
        max.mean <- max(sigma.u.mean, sigma.e.mean)
        min.mean <- min(sigma.u.mean, sigma.e.mean)
        mean.range <- max.mean - min.mean
        upper.limit.mean <- max.mean + 0.05*mean.range
        lower.limit.mean <- min.mean - 0.2*mean.range

        plot(x=thinned.iterations,
             y=sigma.e.mean,
             type="l",
             main=overlay.mean.title,
             xlab="Number of Iterations",
             ylab="Parameter Mean",
             ylim=c(lower.limit.mean, upper.limit.mean))
        lines(x=thinned.iterations,
              y=sigma.u.mean,
              col="blue")
        legend(x="bottomright",
               legend=c("Sigma-e", "Sigma-u"),
               col=c("black", "blue"),
               lty=c(1,1),
               lwd=c(2,2),
               bty="n")
      }
    }
  }

  return(NULL)
}

#Produces trace plots of alpha1 values and means.
alpha1_trace_plots <- function(alpha1,
                               num.iterations,
                               num.burn,
                               num.thin,
                               plot.type) {

  #alpha1 trace plots
  never.consumer.covariates <- rownames(alpha1)
  for(covar.h in seq_along(never.consumer.covariates)) {

    alpha1.value <- alpha1[covar.h,]

    if("value" %in% plot.type) {

      value.plot.title <- paste0("Alpha1 Values, ", never.consumer.covariates[covar.h])
      plot(x=seq_along(alpha1.value),
           y=alpha1.value,
           type="l",
           main=value.plot.title,
           xlab="Number of Iterations",
           ylab="Alpha1 Value")
      abline(v=num.burn, col="red")
    }

    if("mean" %in% plot.type) {

      thinned.iterations <- seq(num.burn+1, num.iterations, num.thin)
      alpha1.thinned <- alpha1.value[thinned.iterations]
      alpha1.mean <- cumsum(alpha1.thinned)/seq_along(alpha1.thinned)

      mean.plot.title <- paste0("Alpha1 Means, ", never.consumer.covariates[covar.h])
      plot(x=thinned.iterations,
           y=alpha1.mean,
           type="l",
           main=mean.plot.title,
           xlab="Number of Iterations",
           ylab="Alpha1 Mean")
    }
  }

  return(NULL)
}

#Produces trace plot of average consumer probabilities for the first episodic food.
consumer_probability_trace_plots <- function(consumer.probabilities,
                                             num.iterations,
                                             num.burn,
                                             num.thin,
                                             plot.type) {

  consumer.probability.value <- consumer.probabilities

  if("value" %in% plot.type) {

    value.plot.title <- "Consumer Probability Values"
    plot(x=seq_along(consumer.probability.value),
         y=consumer.probability.value,
         type="l",
         main=value.plot.title,
         xlab="Number of Iterations",
         ylab="Consumer Probability Value")
    abline(v=num.burn, col="red")
  }

  if("mean" %in% plot.type) {

    thinned.iterations <- seq(num.burn+1, num.iterations, num.thin)
    consumer.probability.thinned <- consumer.probability.value[thinned.iterations]
    consumer.probability.mean <- cumsum(consumer.probability.thinned)/seq_along(consumer.probability.thinned)

    mean.plot.title <- "Consumer Probability Means"
    plot(x=thinned.iterations,
         y=consumer.probability.mean,
         type="l",
         main=mean.plot.title,
         xlab="Number of Iterations",
         ylab="Consumer Probability Mean")
  }

  return(NULL)
}
