#' Extract NCI Multivar MCMC Parameters
#'
#' @description Extract posterior means and covariances from an
#'   `nci.multivar.mcmc` object.
#'
#' @details This function calculates the posterior means and covariances of
#'   parameters using the MCMC samples in an `nci.multivar.mcmc` object. Means
#'   and covariances are calculated for the fixed effect coefficients ('beta'),
#'   random effect covariance ('Sigma-u'), and residual error covariance
#'   ('Sigma-e') for all models. For models allowing never-consumers, the
#'   never-consumer model coefficients ('alpha1') and mean consumer probability
#'   are also included. The calculations use samples after the burn-in period,
#'   spaced apart by the thinning number. It is important to note that the
#'   covariance matrix returned by this function is the covariance matrix of the
#'   Monte Carlo process, and it is distinct from the covariances that would be
#'   found from replication techniques such as bootstrap or BRR.
#'
#' @param multivar.mcmc.model An `nci.multivar.mcmc` object.
#'
#' @returns A list with the following elements:
#' * mean: A named vector of posterior means for each MCMC parameter.
#' * covariance: A matrix of variances and covariances for each pair of MCMC parameters.
#'
#' @export
#'
#' @examples
#' #subset NHANES data
#' nhanes.subset <- nhcvd[nhcvd$SDMVSTRA %in% c(48, 60, 72),]
#'
#' boxcox.lambda.data <- boxcox_survey(input.data=nhanes.subset,
#'                                     row.subset=(nhanes.subset$DAY == 1),
#'                                     variable="TSODI",
#'                                     id="SEQN",
#'                                     repeat.obs="DAY",
#'                                     weight="WTDRD1",
#'                                     covariates="RIDAGEYR")
#'
#' minimum.amount.data <- calculate_minimum_amount(input.data=nhanes.subset,
#'                                                 row.subset=(nhanes.subset$DAY == 1),
#'                                                 daily.variables="TSODI")
#'
#' pre.mcmc.data <- nci_multivar_preprocessor(input.data=nhanes.subset,
#'                                            daily.variables="TSODI",
#'                                            continuous.covariates="RIDAGEYR",
#'                                            boxcox.lambda.data=boxcox.lambda.data,
#'                                            minimum.amount.data=minimum.amount.data)
#'
#' mcmc.output <- nci_multivar_mcmc(pre.mcmc.data=pre.mcmc.data,
#'                                  id="SEQN",
#'                                  weight="WTDRD1",
#'                                  repeat.obs="DAY",
#'                                  daily.variables="TSODI",
#'                                  default.covariates="std.RIDAGEYR",
#'                                  num.mcmc.iterations=1000,
#'                                  num.burn=500,
#'                                  num.thin=1)
#'
#' mcmc.parameters <- extract_parameters(mcmc.output)
#'
#' #posterior means
#' mcmc.parameters$mean
#'
#' #posterior covariance matrix
#' mcmc.parameters$covariance
extract_parameters <- function(multivar.mcmc.model) {

  #sample thinned iterations after burn-in
  num.mcmc.iterations <- multivar.mcmc.model$num.mcmc.iterations
  num.burn <- multivar.mcmc.model$num.burn
  num.thin <- multivar.mcmc.model$num.thin

  sampled.iterations <- seq(num.burn+1, num.mcmc.iterations, num.thin)
  num.samples <- (num.mcmc.iterations - (num.burn + 1)) %/% num.thin + 1

  #list of variables in the order that they appear in the model parameters
  all.variables <- c(rbind(multivar.mcmc.model$episodic.indicators, multivar.mcmc.model$episodic.amounts), multivar.mcmc.model$daily.amounts)
  num.variables <- length(all.variables)

  #extract samples of beta, Sigma-e, and Sigma-u with burn-in and thinned iterations discarded
  #only half of the Sigma-e and Sigma-u matrices need to be saved since they are symmetrical
  beta.samples <- vector(mode="list", length=num.variables)
  for(var.j in seq_len(num.variables)) {

    num.beta <- nrow(multivar.mcmc.model$beta[[var.j]])
    beta.samples[[var.j]] <- matrix(nrow=num.beta, ncol=num.samples)
    for(covar.j in seq_len(num.beta)) {

      beta.samples[[var.j]][covar.j,] <- multivar.mcmc.model$beta[[var.j]][covar.j, sampled.iterations]
    }
    rownames(beta.samples[[var.j]]) <- paste("beta", all.variables[var.j], rownames(multivar.mcmc.model$beta[[var.j]]), sep=".")
  }

  num.sigma <- sum(seq_len(num.variables))
  sigma.e.samples <- matrix(nrow=num.sigma, ncol=num.samples)
  sigma.u.samples <- matrix(nrow=num.sigma, ncol=num.samples)
  index <- 0
  for(row.num in seq_len(num.variables)) {

    for(col.num in seq_len(row.num)) {

      index <- index + 1
      sigma.e.samples[index,] <- multivar.mcmc.model$sigma.e[row.num, col.num, sampled.iterations]
      sigma.u.samples[index,] <- multivar.mcmc.model$sigma.u[row.num, col.num, sampled.iterations]
    }
  }
  rownames(sigma.e.samples) <- paste("sigma.e", all.variables[rep(1:num.variables, 1:num.variables)], all.variables[sequence(1:num.variables)], sep=".")
  rownames(sigma.u.samples) <- paste("sigma.u", all.variables[rep(1:num.variables, 1:num.variables)], all.variables[sequence(1:num.variables)], sep=".")

  if(multivar.mcmc.model$never.consumers.first.episodic) {

    num.alpha1 <- nrow(multivar.mcmc.model$alpha1)
    alpha1.samples <- matrix(nrow=num.alpha1, ncol=num.samples)
    for(covar.h in seq_len(num.alpha1)) {

      alpha1.samples[covar.h,] <- multivar.mcmc.model$alpha1[covar.h, sampled.iterations]
    }
    rownames(alpha1.samples) <- paste("alpha", all.variables[1], rownames(multivar.mcmc.model$alpha1), sep=".")

    consumer.probabilities.samples <- matrix(nrow=1, ncol=num.samples)
    consumer.probabilities.samples[1,] <- multivar.mcmc.model$consumer.probabilities[sampled.iterations]
    rownames(consumer.probabilities.samples) <- paste("consumer.probability", all.variables[1], sep=".")

  } else {

    alpha1.samples <- NULL
    consumer.probabilities.samples <- NULL
  }

  parameter.samples <- rbind(do.call(rbind, beta.samples), sigma.e.samples, sigma.u.samples, alpha1.samples, consumer.probabilities.samples)

  #calculate means and variance/covariance matrix
  parameter.means <- rowMeans(parameter.samples)
  parameter.covariance <- ((parameter.samples - parameter.means) %*% t(parameter.samples - parameter.means))/(num.samples - 1)

  parameter.output <- list(mean=parameter.means,
                           covariance=parameter.covariance)

  return(parameter.output)
}

#' Modify Burn-In and Thinning for MCMC Model
#'
#' @description This function modifies the burn-in and thinning values of an
#'   `nci.multivar.mcmc` object that are used to calculate posterior means and
#'   covariances. This can be useful for determining the impact of different
#'   values and finding optimal numbers that balance accuracy and computation
#'   speed.
#'
#' @details Since `nci.multivar.mcmc` objects save MCMC samples from every
#'   iteration, the number of iterations, burn-in, and thinning values used to
#'   calculate posterior means and covariances can be changed. This utility can
#'   be used to update those values while performing error-checking.
#'
#'   For models with stored random effect ('u') matrices from the main MCMC
#'   chain, not all 'u' matrices are stored by default. As a result, the burn-in
#'   for models with main MCMC chain 'u' matrices cannot be decreased, and the
#'   new thinning value must be a multiple of the original thinning value. An
#'   error will be thrown if 'u' matrices are not found for the new thinned
#'   iterations.
#'
#'   Since changing the number of iterations, burn-in, and thinning can alter
#'   the posterior means, post-MCMC 'u' matrix draws conditional on the
#'   posterior mean may not be valid and will throw a warning.
#'
#'
#' @param multivar.mcmc.model An `nci.multivar.mcmc` object.
#' @param num.mcmc.iterations An integer specifying the new total number of
#'   iterations in the MCMC chain, including burn-in.
#' @param num.burn An integer specifying the new number of burn-in iterations in
#'   the MCMC chain.
#' @param num.thin An integer specifying the new number of iterations between
#'   MCMC samples used for calculating posterior means.
#'
#' @returns An `nci.multivar.mcmc` object with the total number of MCMC
#'   iterations, burn-in iterations, and thinning value updated to the new
#'   values.
#'
#' @export
#'
#' @examples
#' #subset NHANES data
#' nhanes.subset <- nhcvd[nhcvd$SDMVSTRA %in% c(48, 60, 72),]
#'
#' boxcox.lambda.data <- boxcox_survey(input.data=nhanes.subset,
#'                                     row.subset=(nhanes.subset$DAY == 1),
#'                                     variable="TSODI",
#'                                     id="SEQN",
#'                                     repeat.obs="DAY",
#'                                     weight="WTDRD1",
#'                                     covariates="RIDAGEYR")
#'
#' minimum.amount.data <- calculate_minimum_amount(input.data=nhanes.subset,
#'                                                 row.subset=(nhanes.subset$DAY == 1),
#'                                                 daily.variables="TSODI")
#'
#' pre.mcmc.data <- nci_multivar_preprocessor(input.data=nhanes.subset,
#'                                            daily.variables="TSODI",
#'                                            continuous.covariates="RIDAGEYR",
#'                                            boxcox.lambda.data=boxcox.lambda.data,
#'                                            minimum.amount.data=minimum.amount.data)
#'
#' mcmc.output <- nci_multivar_mcmc(pre.mcmc.data=pre.mcmc.data,
#'                                  id="SEQN",
#'                                  weight="WTDRD1",
#'                                  repeat.obs="DAY",
#'                                  daily.variables="TSODI",
#'                                  default.covariates="std.RIDAGEYR",
#'                                  num.mcmc.iterations=1000,
#'                                  num.burn=500,
#'                                  num.thin=1)
#'
#' #original iteration numbers
#' mcmc.output$num.mcmc.iterations
#' mcmc.output$num.burn
#' mcmc.output$num.thin
#'
#' mcmc.output <- burn_and_thin(multivar.mcmc.model=mcmc.output,
#'                              num.burn=200,
#'                              num.thin=2)
#'
#' #new iteration numbers
#' mcmc.output$num.mcmc.iterations
#' mcmc.output$num.burn
#' mcmc.output$num.thin
burn_and_thin <- function(multivar.mcmc.model,
                          num.mcmc.iterations=NULL,
                          num.burn=NULL,
                          num.thin=NULL) {

  if(is.null(num.mcmc.iterations)) {

    num.mcmc.iterations <- multivar.mcmc.model$num.mcmc.iterations
  }

  if(is.null(num.burn)) {

    num.burn <- multivar.mcmc.model$num.burn
  }

  if(is.null(num.thin)) {

    num.thin <- multivar.mcmc.model$num.thin
  }

  if(num.mcmc.iterations > multivar.mcmc.model$num.trace) {

    stop("Number of MCMC iterations cannot be higher than the number of traces in the model")
  }

  if(num.burn >= num.mcmc.iterations) {

    stop("Number of MCMC iterations must be greater than the number of burn-in iterations")
  }

  if(!is.null(multivar.mcmc.model$saved.u.main)) {

    if(!all(seq(num.burn+1, num.mcmc.iterations, num.thin) %in% multivar.mcmc.model$saved.u.main)) {

      stop("MCMC chain U matrices not saved for requested iterations")
    }
  }

  if(multivar.mcmc.model$num.post > 0) {

    warning("Post-MCMC U matrices may not be valid because posterior means may have changed")
  }

  #Set new total iterations, burn-in, and thinning
  multivar.mcmc.model$num.mcmc.iterations <- num.mcmc.iterations
  multivar.mcmc.model$num.burn <- num.burn
  multivar.mcmc.model$num.thin <- num.thin

  return(multivar.mcmc.model)
}

#' Draw Parameter Samples from an MCMC Chain
#'
#' @description This function extracts samples of model parameters from an MCMC
#'   chain at a specified interval.
#'
#' @details Drawing parameter values from an MCMC chain after the burn-in period
#'   is equivalent to drawing from the parameters' posterior distribution. The
#'   variation in the MCMC chain is used to sample different parameter values as
#'   opposed to finding a covariance matrix. Using the MCMC chain has the
#'   advantage of using the actual posterior distribution of the parameters
#'   instead of making an assumption. Accurately sampling parameter values is
#'   required for measurement error correction methods such as multiple
#'   imputation.
#'
#'   The function starts by drawing the parameter set from the first iteration
#'   after burn-in, and then draws parameter sets every `draw.spacing`
#'   iterations until `num.draws` sets have been drawn. If random effects ('u')
#'   matrices from the main MCMC chain are saved, `draw.spacing` must be a
#'   multiple of the original thinning value of `multivar.mcmc.model`.
#'
#'
#' @param multivar.mcmc.model An `nci.multivar.mcmc` object.
#' @param num.draws An integer specifying the number of parameter samples to
#'   draw from the MCMC chain. (default = `10`)
#' @param draw.spacing An integer specifying the number of iterations between
#'   samples. If `NULL`, defaults to 5 times the thinning value of
#'   `multivar.mcmc.model`. (default = `NULL`)
#'
#' @returns A list of parameter samples as separate `nci.multivar.mcmc` objects.
#'
#' @export
#'
#' @examples
#' #subset NHANES data
#' nhanes.subset <- nhcvd[nhcvd$SDMVSTRA %in% c(50, 51, 52),]
#'
#' boxcox.lambda.data <- boxcox_survey(input.data=nhanes.subset,
#'                                     row.subset=(nhanes.subset$DAY == 1),
#'                                     variable="TSODI",
#'                                     id="SEQN",
#'                                     repeat.obs="DAY",
#'                                     weight="WTDRD1",
#'                                     covariates="RIDAGEYR")
#'
#' minimum.amount.data <- calculate_minimum_amount(input.data=nhanes.subset,
#'                                                 row.subset=(nhanes.subset$DAY == 1),
#'                                                 daily.variables="TSODI")
#'
#' pre.mcmc.data <- nci_multivar_preprocessor(input.data=nhanes.subset,
#'                                            daily.variables="TSODI",
#'                                            continuous.covariates="RIDAGEYR",
#'                                            boxcox.lambda.data=boxcox.lambda.data,
#'                                            minimum.amount.data=minimum.amount.data)
#'
#' mcmc.output <- nci_multivar_mcmc(pre.mcmc.data=pre.mcmc.data,
#'                                  id="SEQN",
#'                                  weight="WTDRD1",
#'                                  repeat.obs="DAY",
#'                                  daily.variables="TSODI",
#'                                  default.covariates="std.RIDAGEYR",
#'                                  num.mcmc.iterations=1000,
#'                                  num.burn=500,
#'                                  num.thin=1)
#'
#' parameter.samples <- draw_parameters(multivar.mcmc.model=mcmc.output,
#'                                      num.draws=10)
#'
#' #first three parameter samples
#' parameter.samples[[1]]$beta
#' parameter.samples[[1]]$sigma.u
#' parameter.samples[[1]]$sigma.e
#'
#' parameter.samples[[2]]$beta
#' parameter.samples[[2]]$sigma.u
#' parameter.samples[[2]]$sigma.e
#'
#' parameter.samples[[3]]$beta
#' parameter.samples[[3]]$sigma.u
#' parameter.samples[[3]]$sigma.e
draw_parameters <- function(multivar.mcmc.model,
                            num.draws=10,
                            draw.spacing=NULL) {

  num.mcmc.iterations <- multivar.mcmc.model$num.mcmc.iterations
  num.burn <- multivar.mcmc.model$num.burn
  num.thin <- multivar.mcmc.model$num.thin

  #default thinning for parameter draws to 5 times original thinning number
  if(is.null(draw.spacing)) {

    draw.spacing <- 5*num.thin
  }

  #Check if enough iterations exist for parameter draws
  first.draw <- num.burn + 1
  last.draw <- first.draw + draw.spacing*(num.draws - 1)

  if(last.draw > num.mcmc.iterations) {

    stop("Not enough iterations after burn-in to make the requested number of parameter draws")
  }

  #Check if selected U matrices were saved
  if(!is.null(multivar.mcmc.model$saved.u.main)) {

    if(!all(seq(first.draw, last.draw, draw.spacing) %in% multivar.mcmc.model$saved.u.main)) {

      stop("MCMC chain U matrices not saved for requested draws")
    }
  }

  parameter.draws <- vector(mode="list", length=num.draws)
  for(i in seq_len(num.draws)) {

    draw <- first.draw + draw.spacing*(i-1)

    #draw parameters from specified iteration
    beta.draw <- vector(mode="list", length=length(multivar.mcmc.model$beta))
    for(var.j in seq_along(multivar.mcmc.model$beta)) {

      beta.draw[[var.j]] <- multivar.mcmc.model$beta[[var.j]][,draw,drop=FALSE]
    }
    names(beta.draw) <- names(multivar.mcmc.model$beta)

    sigma.e.draw <- multivar.mcmc.model$sigma.e[,,draw,drop=FALSE]
    sigma.u.draw <- multivar.mcmc.model$sigma.u[,,draw,drop=FALSE]

    if(!is.null(multivar.mcmc.model$saved.u.main)) {

      u.matrix.draw <- multivar.mcmc.model$u.matrices.main[,,which(multivar.mcmc.model$saved.u.main == draw),drop=FALSE]
    } else {

      u.matrix.draw <- NULL
    }

    if(multivar.mcmc.model$never.consumers.first.episodic) {

      alpha1.draw <- multivar.mcmc.model$alpha1[,draw,drop=FALSE]
      consumer.probabilities.draw <- multivar.mcmc.model$consumer.probabilities[draw]
    } else {

      alpha1.draw <- NULL
      consumer.probabilities.draw <- NULL
    }

    #create new NCI multivar MCMC object
    parameter.draws[[i]] <- structure(list(beta=beta.draw,
                                           sigma.u=sigma.u.draw,
                                           sigma.e=sigma.e.draw,
                                           alpha1=alpha1.draw,
                                           consumer.probabilities=consumer.probabilities.draw,
                                           u.matrices.main=NULL,
                                           u.matrices.post=u.matrix.draw,
                                           mcmc.subjects=multivar.mcmc.model$mcmc.subjects,
                                           subject.weighting=multivar.mcmc.model$subject.weighting,
                                           num.episodic=multivar.mcmc.model$num.episodic,
                                           num.daily=multivar.mcmc.model$num.daily,
                                           num.trace=1,
                                           num.mcmc.iterations=1,
                                           num.burn=0,
                                           num.thin=1,
                                           num.post=as.numeric(!is.null(multivar.mcmc.model$saved.u.main)),
                                           saved.u.main=NULL,
                                           backtransformation=multivar.mcmc.model$backtransformation,
                                           episodic.variables=multivar.mcmc.model$episodic.variables,
                                           episodic.indicators=multivar.mcmc.model$episodic.indicators,
                                           episodic.amounts=multivar.mcmc.model$episodic.amounts,
                                           daily.variables=multivar.mcmc.model$daily.variables,
                                           daily.amounts=multivar.mcmc.model$daily.amounts,
                                           covariates=multivar.mcmc.model$covariates,
                                           intercepts=multivar.mcmc.model$intercepts,
                                           never.consumers.first.episodic=multivar.mcmc.model$never.consumers.first.episodic,
                                           never.consumer.covariates=multivar.mcmc.model$never.consumer.covariates,
                                           never.consumer.intercept=multivar.mcmc.model$never.consumer.intercept),
                                      class="nci.multivar.mcmc")
  }

  return(parameter.draws)
}
