# Main Gibbs sampler loop to fit multivar MCMC model.
mcmc_main_loop <- function(num.mcmc.iterations,
                           num.burn,
                           num.post,
                           num.thin,
                           num.subjects,
                           num.episodic,
                           num.daily,
                           num.recalls,
                           recall.availability,
                           subject.weighting,
                           episodic.indicator.matrices,
                           never.consumers.first.episodic,
                           covariate.matrices,
                           weighted.covariate.matrices,
                           weighted.covariate.squared.sums,
                           never.consumer.covariate.matrix,
                           alpha1.mean.prior,
                           alpha1.covariance.prior,
                           consumer.probabilities.prior,
                           beta.mean.prior,
                           beta.covariance.prior,
                           r.matrix.prior,
                           theta.matrix.prior,
                           v.matrix.prior,
                           sigma.e.prior,
                           sigma.u.prior,
                           u.matrix.prior,
                           w.matrix.prior,
                           xbeta.prior,
                           xbeta.u.prior,
                           sigma.u.constant,
                           save.u.main,
                           save.all.u,
                           do.log.likelihood) {

  #Initialize parameters to their priors
  alpha1 <- alpha1.mean.prior
  consumer.probabilities <- consumer.probabilities.prior
  beta <- beta.mean.prior
  r.matrix <- r.matrix.prior
  theta.matrix <- theta.matrix.prior
  v.matrix <- v.matrix.prior
  sigma.e <- sigma.e.prior
  sigma.u <- sigma.u.prior
  u.matrix <- u.matrix.prior
  w.matrix <- w.matrix.prior
  xbeta <- xbeta.prior
  xbeta.u <- xbeta.u.prior

  #Initialize parameter trace data
  thinned.iterations <- seq(num.burn+1, num.mcmc.iterations, num.thin)
  num.thinned.iterations <- (num.mcmc.iterations - (num.burn + 1)) %/% num.thin + 1

  if(never.consumers.first.episodic) {

    alpha1.trace <- matrix(nrow=length(alpha1), ncol=num.mcmc.iterations)
    rownames(alpha1.trace) <- names(alpha1)

    consumer.probabilities.trace <- numeric(num.mcmc.iterations)
  } else {

    alpha1.trace <- NULL
    consumer.probabilities.trace <- NULL
  }

  beta.trace <- vector(mode="list", length=length(beta))
  for(var.j in seq_along(beta)) {

    beta.trace[[var.j]] <- matrix(nrow=length(beta[[var.j]]), ncol=num.mcmc.iterations)
    rownames(beta.trace[[var.j]]) <- names(beta[[var.j]])
  }
  names(beta.trace) <- names(beta)

  sigma.e.trace <- array(dim=c(nrow(sigma.e), ncol(sigma.e), num.mcmc.iterations))
  dimnames(sigma.e.trace) <- dimnames(sigma.e)

  sigma.u.trace <- array(dim=c(nrow(sigma.u), ncol(sigma.u), num.mcmc.iterations))
  dimnames(sigma.u.trace) <- dimnames(sigma.u)

  #initialize U matrix trace data for main MCMC and post-MCMC iterations
  if(save.u.main) {

    if(save.all.u) {

      u.matrix.main <- array(dim=c(nrow(u.matrix), ncol(u.matrix), num.mcmc.iterations))
      saved.u.main <- seq_len(num.mcmc.iterations)
    } else {

      u.matrix.main <- array(dim=c(nrow(u.matrix), ncol(u.matrix), num.thinned.iterations))
      saved.u.main <- thinned.iterations
    }
    dimnames(u.matrix.main) <- dimnames(u.matrix)
  } else {

    u.matrix.main <- NULL
    saved.u.main <- NULL
  }

  if(num.post > 0) {

    u.matrix.post <- array(dim=c(nrow(u.matrix), ncol(u.matrix), num.post))
    dimnames(u.matrix.post) <- dimnames(u.matrix)
  } else {

    u.matrix.post <- NULL
  }

  #Gibbs sampler loop
  for(iter in 1:(num.mcmc.iterations + num.post)) {

    #update Ni
    conni1 <- update_conni1(never.consumers.first.episodic=never.consumers.first.episodic,
                            alpha1=alpha1,
                            consumer.probabilities=consumer.probabilities,
                            xbeta.u=xbeta.u,
                            never.consumer.covariate.matrix=never.consumer.covariate.matrix,
                            episodic.indicator.matrices=episodic.indicator.matrices,
                            recall.availability=recall.availability,
                            num.subjects=num.subjects,
                            num.recalls=num.recalls)

    #update alpha1
    alpha1 <- update_alpha1(never.consumers.first.episodic=never.consumers.first.episodic,
                            alpha1.mean.prior=alpha1.mean.prior,
                            alpha1.covariance.prior=alpha1.covariance.prior,
                            conni1=conni1,
                            never.consumer.covariate.matrix=never.consumer.covariate.matrix,
                            subject.weighting=subject.weighting)

    #update consumer probabilities
    consumer.probabilities <- update_consumer_probabilities(never.consumers.first.episodic=never.consumers.first.episodic,
                                                            never.consumer.covariate.matrix=never.consumer.covariate.matrix,
                                                            alpha1=alpha1)

    #update W matrix
    w.matrix <- update_w_matrix(w.matrix=w.matrix,
                                xbeta.u=xbeta.u,
                                sigma.e=sigma.e,
                                recall.availability=recall.availability,
                                episodic.indicator.matrices=episodic.indicator.matrices,
                                num.subjects=num.subjects,
                                num.episodic=num.episodic,
                                num.daily=num.daily,
                                num.recalls=num.recalls)

    if(iter <= num.mcmc.iterations) {

      #calculate W-XBeta-U cross-residual sum
      #The W-XBeta-U calculation corresponds to the error (epsilon) term in Equation 3.5 of Zhang, et al. (2011)
      w.cross.residual.sum <- calculate_w_cross_residual_sum(w.matrix=w.matrix,
                                                             xbeta.u=xbeta.u,
                                                             recall.availability=recall.availability,
                                                             subject.weighting=subject.weighting,
                                                             num.episodic=num.episodic,
                                                             num.daily=num.daily,
                                                             num.recalls=num.recalls)

      #update r matrix
      r.matrix <- update_r_matrix(r.matrix=r.matrix,
                                  theta.matrix=theta.matrix,
                                  v.matrix=v.matrix,
                                  w.cross.residual.sum=w.cross.residual.sum,
                                  recall.availability=recall.availability,
                                  subject.weighting=subject.weighting)

      #update theta matrix
      theta.matrix <- update_theta_matrix(theta.matrix=theta.matrix,
                                          r.matrix=r.matrix,
                                          v.matrix=v.matrix,
                                          w.cross.residual.sum=w.cross.residual.sum)

      #update V matrix
      v.matrix <- update_v_matrix(v.matrix=v.matrix,
                                  r.matrix=r.matrix,
                                  theta.matrix=theta.matrix,
                                  w.cross.residual.sum=w.cross.residual.sum,
                                  recall.availability=recall.availability,
                                  subject.weighting=subject.weighting,
                                  num.episodic=num.episodic,
                                  num.daily=num.daily)

      #update sigma-e
      sigma.e <- v.matrix %*% t(v.matrix)

      #update sigma-u
      sigma.u <- update_sigma_u(sigma.u=sigma.u,
                                sigma.u.prior=sigma.u.prior,
                                u.matrix=u.matrix,
                                subject.weighting=subject.weighting,
                                sigma.u.constant=sigma.u.constant,
                                num.subjects=num.subjects)
    }

    #update U matrix
    u.matrix <- update_u_matrix(sigma.u=sigma.u,
                                sigma.e=sigma.e,
                                w.matrix=w.matrix,
                                xbeta=xbeta,
                                recall.availability=recall.availability,
                                num.subjects=num.subjects,
                                num.episodic=num.episodic,
                                num.daily=num.daily,
                                num.recalls=num.recalls,
                                never.consumers.first.episodic=never.consumers.first.episodic,
                                u.matrix=u.matrix,
                                conni1=conni1)

    if(iter <= num.mcmc.iterations) {

      #update beta
      beta <- update_beta(weighted.covariate.matrices=weighted.covariate.matrices,
                          weighted.covariate.squared.sums=weighted.covariate.squared.sums,
                          recall.availability=recall.availability,
                          w.matrix=w.matrix,
                          u.matrix=u.matrix,
                          sigma.e=sigma.e,
                          xbeta=xbeta,
                          beta.mean.prior=beta.mean.prior,
                          beta.covariance.prior=beta.covariance.prior,
                          num.subjects=num.subjects,
                          num.episodic=num.episodic,
                          num.daily=num.daily,
                          num.recalls=num.recalls,
                          never.consumers.first.episodic=never.consumers.first.episodic,
                          conni1=conni1,
                          beta1=beta[[1]],
                          covariate.matrices=covariate.matrices)
    }

    #Saving parameters for this iteration
    if(iter <= num.mcmc.iterations) {

      #Save traces
      if(!is.null(never.consumer.covariate.matrix)) {

        alpha1.trace[,iter] <- alpha1

        consumer.probabilities.trace[iter] <- mean(consumer.probabilities)
      }

      for(var.j in seq_along(beta)) {

        beta.trace[[var.j]][,iter] <- beta[[var.j]]
      }

      sigma.e.trace[,,iter] <- sigma.e

      sigma.u.trace[,,iter] <- sigma.u

      #if specified, save U matrix
      if(save.u.main) {

        if(iter %in% saved.u.main) {

          u.matrix.main[,,which(saved.u.main == iter)] <- u.matrix
        }
      }

    } else {

      #save U matrix (post-MCMC)
      u.matrix.post[,,iter - num.mcmc.iterations] <- u.matrix
    }

    #Calculate posterior means of MCMC parameters
    if(iter == num.mcmc.iterations) {

      #calculate means
      beta.mean <- vector(mode="list", length=length(beta.trace))
      for(var.j in seq_along(beta.trace)) {

        beta.mean[[var.j]] <- rowMeans(beta.trace[[var.j]][,thinned.iterations,drop=FALSE])
      }
      names(beta.mean) <- names(beta.trace)

      sigma.e.mean <- rowMeans(sigma.e.trace[,,thinned.iterations,drop=FALSE], dims=2)

      sigma.u.mean <- rowMeans(sigma.u.trace[,,thinned.iterations,drop=FALSE], dims=2)

      #fix beta, sigma-e, and sigma-u at posterior means
      beta <- beta.mean
      sigma.e <- sigma.e.mean
      sigma.u <- sigma.u.mean
    }

    #update Xbeta
    if(iter <= num.mcmc.iterations) {

      for(day.k in seq_len(num.recalls)) {

        for(var.j in seq_len(2*num.episodic+num.daily)) {

          xbeta[[day.k]][,var.j] <- covariate.matrices[[var.j]][[day.k]] %*% beta[[var.j]]
        }
      }
    }

    #update Xbeta-U
    for(day.k in seq_len(num.recalls)) {

      xbeta.u[[day.k]] <- xbeta[[day.k]] + u.matrix
    }
  }

  # calculate log-likelihood
  log.likelihood <- marginal_likelihood(do.log.likelihood=do.log.likelihood,
                                        w.matrix=w.matrix,
                                        episodic.indicator.matrices=episodic.indicator.matrices,
                                        xbeta=xbeta,
                                        sigma.u=sigma.u,
                                        sigma.e=sigma.e,
                                        recall.availability=recall.availability,
                                        subject.weighting=subject.weighting,
                                        num.subjects=num.subjects,
                                        num.episodic=num.episodic,
                                        num.daily=num.daily,
                                        num.recalls=num.recalls,
                                        never.consumers.first.episodic=never.consumers.first.episodic,
                                        never.consumer.covariate.matrix=never.consumer.covariate.matrix,
                                        alpha1=alpha1)

  #Combine parameter traces and log-likelihood into final MCMC_multivar object
  mcmc.output <- list(alpha1=alpha1.trace,
                      consumer.probabilities=consumer.probabilities.trace,
                      beta=beta.trace,
                      sigma.e=sigma.e.trace,
                      sigma.u=sigma.u.trace,
                      u.matrices.main=u.matrix.main,
                      saved.u.main=saved.u.main,
                      u.matrices.post=u.matrix.post,
                      log.likelihood=log.likelihood)
  return(mcmc.output)
}
