#' Gelman-Rubin Test for MCMC Convergence
#'
#' @description Tests an MCMC model specification for convergence using the
#'   Gelman-Rubin test. A model is fit multiple times with the specified
#'   parameters using different random seeds each time.
#'
#' @details The Gelman-Rubin test works by creating multiple MCMC chains with
#'   different random seeds and calculating the within-chain and between-chain
#'   variation (Gelman and Rubin, 1992). If the model parameters converge, there
#'   should be little to no difference between different chains which will cause
#'   the between-chain variance to fall to zero. This means that convergence can
#'   be assessed by comparing the total variance of the model parameters to the
#'   within-chain variance.
#'
#'   The Gelman-Rubin statistic is the square root of the ratio between the
#'   total variance and the within-chain variance of a parameter. If the
#'   Gelman-Rubin statistic is close to 1, then the model has converged for that
#'   parameter. A cutoff of 1.1 for convergence is suggested by Gelman, et al.
#'   (2004).
#'
#' @inheritParams nci_multivar_mcmc
#' @param num.chains Integer number of MCMC chains to run. (default = 5)
#' @param initial.mcmc.seed Numeric starting seed for the random number
#'   generator. If specified, the seed will be incremented for each MCMC
#'   replicate. If `NULL`, uses a randomly generated integer from -10^7 to 10^7,
#'   exclusive for each replicate. (default = `NULL`)
#'
#' @returns A numeric vector of Gelman-Rubin statistics for each MCMC parameter.
#'
#' @export
#'
#' @examples
#' #' #subset NHANES data
#' nhanes.subset <- nhcvd[nhcvd$SDMVSTRA %in% c(48, 60, 72),]
#'
#' boxcox.lambda.data <- boxcox_survey(input.data=nhanes.subset,
#'                                     row.subset=(nhanes.subset$DAY == 1),
#'                                     variable="TSODI",
#'                                     id="SEQN",
#'                                     repeat.obs="DAY",
#'                                     weight="WTDRD1",
#'                                     covariates="RIDAGEYR")
#'
#' minimum.amount.data <- calculate_minimum_amount(input.data=nhanes.subset,
#'                                                 row.subset=(nhanes.subset$DAY == 1),
#'                                                 daily.variables="TSODI")
#'
#' pre.mcmc.data <- nci_multivar_preprocessor(input.data=nhanes.subset,
#'                                            daily.variables="TSODI",
#'                                            continuous.covariates="RIDAGEYR",
#'                                            boxcox.lambda.data=boxcox.lambda.data,
#'                                            minimum.amount.data=minimum.amount.data)
#'
#' gr.statistics <- gelman_rubin(num.chains=5,
#'                               pre.mcmc.data=pre.mcmc.data,
#'                               id="SEQN",
#'                               weight="WTDRD1",
#'                               repeat.obs="DAY",
#'                               daily.variables="TSODI",
#'                               default.covariates="std.RIDAGEYR",
#'                               num.mcmc.iterations=1000,
#'                               num.burn=500,
#'                               num.thin=1)
#'
#' gr.statistics
gelman_rubin <- function(num.chains=5,
                         pre.mcmc.data,
                         id,
                         weight=NULL,
                         repeat.obs,
                         episodic.variables=NULL,
                         episodic.indicators=NULL,
                         episodic.amounts=NULL,
                         daily.variables=NULL,
                         daily.amounts=NULL,
                         default.covariates=NULL,
                         episodic.indicator.covariates=NULL,
                         episodic.amount.covariates=NULL,
                         daily.amount.covariates=NULL,
                         individual.covariates=list(),
                         default.intercept=TRUE,
                         episodic.indicator.intercept=NULL,
                         episodic.amount.intercept=NULL,
                         daily.amount.intercept=NULL,
                         individual.intercept=list(),
                         never.consumer.variable=NULL,
                         never.consumer.indicator=NULL,
                         never.consumer.amount=NULL,
                         never.consumer.covariates=NULL,
                         never.consumer.intercept=TRUE,
                         initial.mcmc.seed=NULL,
                         num.mcmc.iterations=12000,
                         num.burn=2000,
                         num.thin=25,
                         sigma.u.prior=NULL,
                         sigma.u.constant=FALSE) {

  mean.list <- vector(mode="list", length=num.chains)
  variance.list <- vector(mode="list", length=num.chains)

  mcmc.seed <- initial.mcmc.seed

  for(i in seq_len(num.chains)) {

    #Fit MCMC model
    mcmc.output <- nci_multivar_mcmc(pre.mcmc.data=pre.mcmc.data,
                                     id=id,
                                     weight=weight,
                                     repeat.obs=repeat.obs,
                                     episodic.variables=episodic.variables,
                                     episodic.indicators=episodic.indicators,
                                     episodic.amounts=episodic.amounts,
                                     daily.variables=daily.variables,
                                     daily.amounts=daily.amounts,
                                     default.covariates=default.covariates,
                                     episodic.indicator.covariates=episodic.indicator.covariates,
                                     episodic.amount.covariates=episodic.amount.covariates,
                                     daily.amount.covariates=daily.amount.covariates,
                                     individual.covariates=individual.covariates,
                                     default.intercept=default.intercept,
                                     episodic.indicator.intercept=episodic.indicator.intercept,
                                     episodic.amount.intercept=episodic.amount.intercept,
                                     daily.amount.intercept=daily.amount.intercept,
                                     individual.intercept=individual.intercept,
                                     never.consumer.variable=never.consumer.variable,
                                     never.consumer.indicator=never.consumer.indicator,
                                     never.consumer.amount=never.consumer.amount,
                                     never.consumer.covariates=never.consumer.covariates,
                                     never.consumer.intercept=never.consumer.intercept,
                                     mcmc.seed=mcmc.seed,
                                     num.mcmc.iterations=num.mcmc.iterations,
                                     num.burn=num.burn,
                                     num.thin=num.thin,
                                     sigma.u.prior=sigma.u.prior,
                                     sigma.u.constant=sigma.u.constant)

    #Calculate parameter means and variances
    model.parameters <- extract_parameters(mcmc.output)

    mean.list[[i]] <- model.parameters$mean
    variance.list[[i]] <- diag(model.parameters$covariance)

    #Increment seed if fixed
    if(!is.null(mcmc.seed)) {

      mcmc.seed <- mcmc.seed + 1
    }
  }

  means <- do.call(cbind, mean.list)
  variances <- do.call(cbind, variance.list)

  num.samples <- (num.mcmc.iterations - num.burn) %/% num.thin

  #calculate the joint mean of all chains
  joint.mean <- rowMeans(means)

  #calculate the within-chain, between-chain, and total variances for each parameter
  within.variance <- rowMeans(variances)
  between.variance <- (num.samples/(num.chains - 1))*rowSums((means - joint.mean)^2)
  total.variance <- (num.samples/(num.samples - 1))*within.variance + ((num.chains + 1)/(num.chains*num.samples))*between.variance

  #Calculate the Gelman-Rubin statistics for each parameter
  #if both total variance and within-chain variance are zero, the statistic is one
  gr.statistics <- sqrt(total.variance/within.variance)
  gr.statistics[total.variance == 0 & within.variance == 0] <- 1

  return(gr.statistics)
}
