#Capitalizes first letter string.
capitalize <- function(string) {

  new.string <- paste0(toupper(substr(string, 1, 1)), substr(string, 2, nchar(string)))
  return(new.string)
}

#Checks variables for valid matches in a dataset.
check_variables <- function(logger,
                            section,
                            variables,
                            variable.text,
                            dataset,
                            dataset.text) {

  if(is.character(dataset)) {

    dataset.names <- dataset
  } else if(is.matrix(dataset)) {

    dataset.names <- colnames(dataset)
  } else {

    dataset.names <- names(dataset)
  }

  for(i in seq_along(variables)) {

    exact <- dataset.names[variables[i] == dataset.names]
    case <- dataset.names[toupper(variables[i]) == toupper(dataset.names)]
    if(length(exact) == 0 && length(case) == 0) {

      logger <- update_logger(logger, 3, section, paste0(capitalize(variable.text), " ", variables[i], " was not found in ", tolower(dataset.text)))
    } else if(length(exact) == 0 && length(case) >= 2) {

      logger <- update_logger(logger, 3, section, paste0(capitalize(variable.text), " ", variables[i], " has multiple matches in ", dataset.text, ": ", paste(case, collapse=", ")))
    } else if(length(exact) == 1 && length(case) >= 2) {

      logger <- update_logger(logger, 1, section, paste0(capitalize(variable.text), " ", variables[i], " also matched ", paste(case[!(case %in% exact)], collapse=", "), " in ", dataset.text))
    }
  }

  return(logger)
}

#Checks indicators and amounts for corresponding variables in backtransformation data.
check_backtran <- function(logger,
                           section,
                           variables,
                           backtran) {

  if(is.character(backtran)) {

    backtran.variables <- backtran
  } else {

    backtran.variables <- backtran$variable
  }

  for(i in seq_along(variables)) {

    exact <- backtran.variables[vapply(backtran.variables, function(back) grepl(back, variables[i], ignore.case=FALSE), logical(1))]
    case <- backtran.variables[vapply(backtran.variables, function(back) grepl(back, variables[i], ignore.case=TRUE), logical(1))]
    if(length(exact) == 0 && length(case) == 0) {

      logger <- update_logger(logger, 3, section, paste0(variables[i], " did not match any backtransformation variables"))
    } else if(length(exact) == 0 && length(case) >= 2) {

      logger <- update_logger(logger, 3, section, paste0(variables[i], " matched multiple backtransformation variables: ", paste(case, collapse=", ")))
    } else if(length(exact) == 1 && length(case) >= 2) {

      logger <- update_logger(logger, 1, section, paste0(variables[i], " also matched ", paste(case[!(case %in% exact)], collapse=", "), " in backtransformation variables"))
    }
  }

  return(logger)
}

#Checks number of MCMC iterations, burn-in, and thinning.
check_iterations <- function(logger,
                             num.mcmc.iterations,
                             num.burn,
                             num.thin,
                             num.post) {

  if(is.null(num.mcmc.iterations)) {

    logger <- update_logger(logger, 3, "Iterations", "Number of MCMC iterations not specified")
  }

  if(!is.null(num.mcmc.iterations)) {

    if(num.mcmc.iterations < 1) {

      logger <- update_logger(logger, 3, "Iterations", "Number of MCMC iterations is less than one")
    }
  }

  if(is.null(num.burn)) {

    logger <- update_logger(logger, 3, "Iterations", "Number of burn-in iterations not specified")
  }

  if(!is.null(num.burn)) {

    if(num.burn < 0) {

      logger <- update_logger(logger, 3, "Iterations", "Number of burn-in iterations is negative")
    }
  }

  if(!is.null(num.mcmc.iterations) && !is.null(num.burn)) {

    if(num.mcmc.iterations <= num.burn) {

      logger <- update_logger(logger, 3, "Iterations", "Number of burn-in iterations is not less than total number of MCMC iterations")
    }
  } else {

    logger <- update_logger(logger, 1, "Iterations", "Number of MCMC iterations not compared to number of burn-in iterations because one or both were not specified")
  }

  if(is.null(num.thin)) {

    logger <- update_logger(logger, 3, "Iterations", "Thinning number not specified")
  }

  if(!is.null(num.thin)) {

    if(num.thin < 1) {

      logger <- update_logger(logger, 3, "Iterations", "Thinning number is less than 1")
    }
  }

  if(is.null(num.post)) {

    logger <- update_logger(logger, 3, "Iterations", "Number of post-MCMC iterations not specified")
  }

  if(!is.null(num.post)) {

    if(num.post < 0) {

      logger <- update_logger(logger, 3, "Iterations", "Number of post-MCMC iterations is negative")
    }
  }

  return(logger)
}

#Checks MCMC input data.
check_data <- function(logger,
                       pre.mcmc.data,
                       id,
                       repeat.obs,
                       weight) {

  if(is.null(pre.mcmc.data$mcmc.input)) {

    logger <- update_logger(logger, 3, "Data", "'mcmc.input' not found in 'pre.mcmc.data'")
  }

  if(is.null(id)) {

    logger <- update_logger(logger, 3, "ID", "Subject ID variable not specified")
  }

  if(!is.null(pre.mcmc.data$mcmc.input) && !is.null(id)) {

    logger <- check_variables(logger=logger,
                              section="ID",
                              variables=id,
                              variable.text="subject identifier",
                              dataset=pre.mcmc.data$mcmc.input,
                              dataset.text="input data")
  } else {

    logger <- update_logger(logger, 1, "ID", "ID variable could not be checked for existence in input data due to errors in variable or data")
  }

  if(is.null(repeat.obs)) {

    logger <- update_logger(logger, 3, "Repeat", "Repeat observation variable not specified")
  }

  if(!is.null(pre.mcmc.data$mcmc.input) & !is.null(repeat.obs)) {

    logger <- check_variables(logger=logger,
                              section="Repeat",
                              variables=repeat.obs,
                              variable.text="repeat observation variable",
                              dataset=pre.mcmc.data$mcmc.input,
                              dataset.text="input data")
  } else {

    logger <- update_logger(logger, 1, "Repeat", "Repeat observation variable could not be checked for existence in input data due to errors in variable or data")
  }

  if(!is.null(weight)) {

    if(!is.null(pre.mcmc.data$mcmc.input)) {

      logger <- check_variables(logger=logger,
                                section="Weight",
                                variables=weight,
                                variable.text="weight variable",
                                dataset=pre.mcmc.data$mcmc.input,
                                dataset.text="input data")
    } else {

      logger <- update_logger(logger, 1, "Weight", "Weight variable could not be checked for existence in input data due to errors in input data")
    }
  }



  if(is.null(pre.mcmc.data$backtransformation)) {

    logger <- update_logger(logger, 3, "Backtran", "'backtransformation' not found in 'pre.mcmc.data'")
  }

  if(!is.null(pre.mcmc.data$backtransformation)) {

    logger <- check_variables(logger=logger,
                              section="Backtran",
                              variables=c("variable", "tran_lambda", "minamount", "tran_center", "tran_scale", "biomarker"),
                              variable.text="variable",
                              dataset=pre.mcmc.data$backtransformation,
                              dataset.text="backtransformation data")
  } else {

    logger <- update_logger(logger, 1, "Backtran", "Variables could not be checked for existence in backtransformation data due to errors in backtransformation data")
  }

  return(logger)
}

#Checks variable list parameters.
check_variable_lists <- function(logger,
                                 pre.mcmc.data,
                                 episodic.variables,
                                 episodic.indicators,
                                 episodic.amounts,
                                 daily.variables,
                                 daily.amounts,
                                 never.consumer.variable,
                                 never.consumer.indicator,
                                 never.consumer.amount) {

  is.nci.multivar <- is(pre.mcmc.data, "nci.multivar.preprocessor")
  data.checks.passed <- all(logger$record$error.code[logger$record$section == "Data"] != 3)
  backtran.checks.passed <- all(logger$record$error.code[logger$record$section == "Backtran"] != 3)

  #Episodic variables
  if(length(episodic.variables) > 0) {

    if(backtran.checks.passed) {

      logger <- check_variables(logger=logger,
                                section="Variables",
                                variables=episodic.variables,
                                variable.text="episodic variable",
                                dataset=pre.mcmc.data$backtransformation$variable,
                                dataset.text="backtransformation variables")
    } else {

      logger <- update_logger(logger, 1, "Variables", "Episodic variables could not be checked for existence in the backtransformation variables due to errors in backtransformation data")
    }
  }

  if(length(episodic.variables) > 0 && length(episodic.indicators) == 0) {

    if(!is.nci.multivar) {

      logger <- update_logger(logger, 3, "Variables", "Episodic indicators not specified with episodic variables when nci_multivar_preprocessor() was not used")
    }
  }

  if(length(episodic.variables) > 0 && length(episodic.amounts) == 0) {

    if(!is.nci.multivar) {

      logger <- update_logger(logger, 3, "Variables", "Episodic amounts not specified with episodic variables when nci_multivar_preprocessor() was not used")
    }
  }

  if(length(episodic.variables) == 0 && length(episodic.indicators) > 0) {

    if(backtran.checks.passed) {

      logger <- check_backtran(logger=logger,
                               section="Variables",
                               variables=episodic.indicators,
                               backtran=pre.mcmc.data$backtransformation)
    } else {

      logger <- update_logger(logger, 1, "Variables", "Episodic indicators could not be checked for correspondence with backtransformation variables due to errors in backtransformation data")
    }
  }

  if(length(episodic.variables) == 0 && length(episodic.amounts) > 0) {

    if(backtran.checks.passed) {

      logger <- check_backtran(logger=logger,
                               section="Variables",
                               variables=episodic.amounts,
                               backtran=pre.mcmc.data$backtransformation)
    } else {

      logger <- update_logger(logger, 1, "Variables", "Episodic amounts could not be checked for correspondence with backtransformation variables due to errors in backtransformation data")
    }
  }

  if(length(episodic.indicators) > 0) {

    if(data.checks.passed) {

      logger <- check_variables(logger=logger,
                                section="Variables",
                                variables=episodic.indicators,
                                variable.text="episodic indicator",
                                dataset=pre.mcmc.data$mcmc.input,
                                dataset.text="input data")
    } else {

      logger <- update_logger(logger, 1, "Variables", "Episodic indicators could not be checked for existence in input data due to errors in input data")
    }
  }

  if(length(episodic.amounts) > 0) {

    if(data.checks.passed) {

      logger <- check_variables(logger=logger,
                                section="Variables",
                                variables=episodic.amounts,
                                variable.text="episodic amount",
                                dataset=pre.mcmc.data$mcmc.input,
                                dataset.text="input data")
    } else {

      logger <- update_logger(logger, 1, "Variables", "Episodic amounts could not be checked for existence in input data due to errors in input data")
    }

  }

  #Daily variables
  if(length(daily.variables) > 0) {

    if(backtran.checks.passed) {

      logger <- check_variables(logger=logger,
                                section="Variables",
                                variables=daily.variables,
                                variable.text="daily variable",
                                dataset=pre.mcmc.data$backtransformation$variable,
                                dataset.text="backtransformation variables")
    } else {

      logger <- update_logger(logger, 1, "Variables", "Daily variables could not be checked for existence in the backtransformation variables due to errors in backtransformation data")
    }
  }

  if(length(daily.variables) > 0 && length(daily.amounts) == 0) {

    if(!is.nci.multivar) {

      logger <- update_logger(logger, 3, "Variables", "Daily amounts not specified with daily variables when nci_multivar_preprocessor() was not used")
    }
  }

  if(length(daily.variables) == 0 && length(daily.amounts) > 0) {

    if(backtran.checks.passed) {

      logger <- check_backtran(logger=logger,
                               section="Variables",
                               variables=daily.amounts,
                               backtran=pre.mcmc.data$backtransformation)
    } else {

      logger <- update_logger(logger, 1, "Variables", "Episodic amounts could not be checked for correspondence with backtransformation variables due to errors in backtransformation data")
    }
  }

  if(length(daily.amounts) > 0) {

    if(data.checks.passed) {

      logger <- check_variables(logger=logger,
                                section="Variables",
                                variables=daily.amounts,
                                variable.text="daily amount",
                                dataset=pre.mcmc.data$mcmc.input,
                                dataset.text="input data")
    } else {

      logger <- update_logger(logger, 1, "Variables", "Daily amounts could not be checked for existence in input data due to errors in input data")
    }
  }

  #Never-consumer variable
  if(length(never.consumer.variable) > 0) {

    if(backtran.checks.passed) {

      logger <- check_variables(logger=logger,
                                section="Variables",
                                variables=never.consumer.variable,
                                variable.text="never-consumer variable",
                                dataset=pre.mcmc.data$backtransformation$variable,
                                dataset.text="backtransformation variables")
    } else {

      logger <- update_logger(logger, 1, "Variables", "Never-consumer variable could not be checked for existence in the backtransformation variables due to errors in backtransformation data")
    }
  }

  if(length(never.consumer.variable) > 1) {

    logger <- update_logger(logger, 3, "Variables", "More than one never-consumer variable specified")
  }

  if(length(never.consumer.variable) > 0 && length(never.consumer.indicator) == 0) {

    if(!is.nci.multivar) {

      logger <- update_logger(logger, 3, "Variables", "Never-consumer indicator not specified with never-consumer variable when nci_multivar_preprocessor() was not used")
    }
  }

  if(length(never.consumer.variable) > 0 && length(never.consumer.amount) == 0) {

    if(!is.nci.multivar) {

      logger <- update_logger(logger, 3, "Variables", "Never-consumer amount not specified with never-consumer variable when nci_multivar_preprocessor() was not used")
    }
  }

  if(length(never.consumer.variable) == 0 && length(never.consumer.indicator) > 0) {

    if(backtran.checks.passed) {

      logger <- check_backtran(logger=logger,
                               section="Variables",
                               variables=never.consumer.indicator,
                               backtran=pre.mcmc.data$backtransformation)
    } else {

      logger <- update_logger(logger, 1, "Variables", "Never-consumer indicator could not be checked for correspondence with backtransformation variables due to errors in backtransformation data")
    }
  }

  if(length(never.consumer.variable) == 0 && length(never.consumer.amount) > 0) {

    if(backtran.checks.passed) {

      logger <- check_backtran(logger=logger,
                               section="Variables",
                               variables=never.consumer.amount,
                               backtran=pre.mcmc.data$backtransformation)
    } else {

      logger <- update_logger(logger, 1, "Variables", "Never-consumer amount could not be checked for correspondence with backtransformation variables due to errors in backtransformation data")
    }
  }

  if(length(never.consumer.indicator) > 0) {

    if(data.checks.passed) {

      logger <- check_variables(logger=logger,
                                section="Variables",
                                variables=never.consumer.indicator,
                                variable.text="never-consumer indicator",
                                dataset=pre.mcmc.data$mcmc.input,
                                dataset.text="input data")
    } else {

      logger <- update_logger(logger, 1, "Variables", "Never-consumer indicator could not be checked for existence in input data due to errors in input data")
    }
  }

  if(length(never.consumer.indicator) > 1) {

    logger <- update_logger(logger, 3, "Variables", "More than one never-consumer indicator specified")
  }

  if(length(never.consumer.amount) > 0) {

    if(data.checks.passed) {

      logger <- check_variables(logger=logger,
                                section="Variables",
                                variables=never.consumer.amount,
                                variable.text="never-consumer amount",
                                dataset=pre.mcmc.data$mcmc.input,
                                dataset.text="input data")
    } else {

      logger <- update_logger(logger, 1, "Variables", "Never-consumer amount could not be checked for existence in input data due to errors in input data")
    }
  }

  if(length(never.consumer.amount) > 1) {

    logger <- update_logger(logger, 3, "Variables", "More than one never-consumer amount specified")
  }

  return(logger)
}

#Checks covariate list parameters.
check_covariate_lists <- function(logger,
                                  pre.mcmc.data,
                                  default.covariates,
                                  episodic.indicator.covariates,
                                  episodic.amount.covariates,
                                  daily.amount.covariates,
                                  individual.covariates,
                                  never.consumer.covariates,
                                  episodic.variables,
                                  episodic.indicators,
                                  episodic.amounts,
                                  daily.variables,
                                  daily.amounts,
                                  never.consumer.variable,
                                  never.consumer.indicator,
                                  never.consumer.amount) {

  is.nci.multivar <- is(pre.mcmc.data, "nci.multivar.preprocessor")
  data.checks.passed <- all(logger$record$error.code[logger$record$section == "Data"] != 3)
  variable.checks.passed <- all(logger$record$error.code[logger$record$section == "Variables"] != 3)

  if(length(default.covariates) > 0) {

    if(data.checks.passed) {

      logger <- check_variables(logger=logger,
                                section="Covariates",
                                variables=default.covariates,
                                variable.text="default covariate",
                                dataset=pre.mcmc.data$mcmc.input,
                                dataset.text="input data")
    } else {

      logger <- update_logger(logger, 1, "Covariates", "Default covariate existence in input data cannot be checked due to errors in input data")
    }
  }

  if(length(episodic.indicator.covariates) > 0) {

    if(data.checks.passed) {

      logger <- check_variables(logger=logger,
                                section="Covariates",
                                variables=episodic.indicator.covariates,
                                variable.text="episodic indicator covariate",
                                dataset=pre.mcmc.data$mcmc.input,
                                dataset.text="input data")
    } else {

      logger <- update_logger(logger, 1, "Covariates", "Episodic indicator covariate existence in input data cannot be checked due to errors in input data")
    }
  }

  if(length(episodic.amount.covariates) > 0) {

    if(data.checks.passed) {

      logger <- check_variables(logger=logger,
                                section="Covariates",
                                variables=episodic.amount.covariates,
                                variable.text="episodic amount covariate",
                                dataset=pre.mcmc.data$mcmc.input,
                                dataset.text="input data")
    } else {

      logger <- update_logger(logger, 1, "Covariates", "Episodic amount covariate existence in input data cannot be checked due to errors in input data")
    }
  }

  if(length(daily.amount.covariates) > 0) {

    if(data.checks.passed) {

      logger <- check_variables(logger=logger,
                                section="Covariates",
                                variables=daily.amount.covariates,
                                variable.text="daily amount covariate",
                                dataset=pre.mcmc.data$mcmc.input,
                                dataset.text="input data")
    } else {

      logger <- update_logger(logger, 1, "Covariates", "Daily amount covariate existence in input data cannot be checked due to errors in input data")
    }
  }

  if(length(individual.covariates) > 0) {

    named.list <- is.list(individual.covariates) && !is.null(names(individual.covariates))
    if(!named.list) {

      logger <- update_logger(logger, 3, "Covariates", "Individual variable covariates are not specified as a named list")
    }

    if(named.list && variable.checks.passed) {

      #process indicator and amount lists
      #this is needed to account for different ways to specify the variable lists
      variable.list <- process_variable_lists(pre.mcmc.data=pre.mcmc.data,
                                              episodic.variables=episodic.variables,
                                              episodic.indicators=episodic.indicators,
                                              episodic.amounts=episodic.amounts,
                                              daily.variables=daily.variables,
                                              daily.amounts=daily.amounts,
                                              never.consumer.variable=never.consumer.variable,
                                              never.consumer.indicator=never.consumer.indicator,
                                              never.consumer.amount=never.consumer.amount)
      episodic.indicators <- variable.list$episodic.indicators
      episodic.amounts <- variable.list$episodic.amounts
      daily.amounts <- variable.list$daily.amounts

      logger <- check_variables(logger=logger,
                                section="Covariates",
                                variables=names(individual.covariates),
                                variable.text="covariate list name",
                                dataset=c(episodic.indicators, episodic.amounts, daily.amounts),
                                dataset.text="indicators and amounts")
    } else {

      logger <- update_logger(logger, 1, "Covariates", "Individual variable covariate list names could not be checked because they are not correctly specified")
    }

    if(named.list && data.checks.passed) {

      for(var.j in names(individual.covariates)) {

        logger <- check_variables(logger=logger,
                                  section="Covariates",
                                  variables=individual.covariates[[var.j]],
                                  variable.text=paste0("variable ", var.j, " covariate"),
                                  dataset=pre.mcmc.data$mcmc.input,
                                  dataset.text="input data")
      }
    } else {

      logger <- update_logger(logger, 1, "Covariates", "Individual variable covariate existence in input data could not be checked due to errors in input data or covariate specification")
    }
  }

  if(length(never.consumer.covariates) > 0) {

    if(data.checks.passed) {

      logger <- check_variables(logger=logger,
                                section="Covariates",
                                variables=never.consumer.covariates,
                                variable.text="never-consumer covariate",
                                dataset=pre.mcmc.data$mcmc.input,
                                dataset.text="input data")
    } else {

      logger <- update_logger(logger, 1, "Covariates", "Never-consumer covariate existence in input data cannot be checked due to errors in input data")
    }
  }

  return(logger)
}

#Checks Sigma-u prior.
check_sigma_u_prior <- function(logger,
                                sigma.u.prior,
                                pre.mcmc.data,
                                episodic.variables,
                                episodic.indicators,
                                episodic.amounts,
                                daily.variables,
                                daily.amounts,
                                never.consumer.variable,
                                never.consumer.indicator,
                                never.consumer.amount) {

  if(is.null(sigma.u.prior)) {

    return(logger)
  }

  variable.checks.passed <- all(logger$record$error.code[logger$record$section == "Variables"] != 3)

  square <- (nrow(sigma.u.prior) == ncol(sigma.u.prior))
  if(!square) {

    logger <- update_logger(logger, 3, "Priors", "Sigma-u prior is not square")
  }

  if(square) {

    if(variable.checks.passed) {

      variable.list <- process_variable_lists(pre.mcmc.data=pre.mcmc.data,
                                              episodic.variables=episodic.variables,
                                              episodic.indicators=episodic.indicators,
                                              episodic.amounts=episodic.amounts,
                                              daily.variables=daily.variables,
                                              daily.amounts=daily.amounts,
                                              never.consumer.variable=never.consumer.variable,
                                              never.consumer.indicator=never.consumer.indicator,
                                              never.consumer.amount=never.consumer.amount)
      num.variables <- length(variable.list$episodic.indicators) + length(variable.list$episodic.amounts) + length(variable.list$daily.amounts)

      if(nrow(sigma.u.prior) != num.variables) {

        logger <- update_logger(logger, 3, "Priors", paste0("Size of Sigma-u prior (", nrow(sigma.u.prior), ") does not match number of variables (", num.variables, ")"))
      }
    } else {

      logger <- update_logger(logger, 1, "Priors", "Sigma-u prior size could not be checked against number of variables because variables were not specified correctly")
    }
  } else {

    logger <- update_logger(logger, 1, "Priors", "Sigma-u prior size could not be checked because Sigma-u prior is not square")
  }

  symmetric <- square && all(abs(sigma.u.prior - t(sigma.u.prior)) < sqrt(.Machine$double.eps))
  if(!symmetric) {

    logger <- update_logger(logger, 3, "Priors", "Sigma-u prior is not symmetric")
  }

  if(symmetric) {

    eigvals <- eigen(sigma.u.prior, symmetric=TRUE, only.values=TRUE)$values
    if(any(eigvals < sqrt(.Machine$double.eps))) {

      logger <- update_logger(logger, 3, "Priors", "Sigma-u prior is not positive definite")
    }
  }

  return(logger)
}

#Checks design matrices.
check_design_matrices <- function(logger,
                                  covariate.matrices,
                                  never.consumers.first.episodic) {

  recall.covariate.matrices <- covariate.matrices$recall
  for(var.j in names(recall.covariate.matrices)) {

    recall.covariate.matrix <- do.call(rbind, recall.covariate.matrices[[var.j]])

    if(ncol(recall.covariate.matrix) == 0) {

      logger <- update_logger(logger, 3, "Design Matrices", paste0("No covariates or intercept for ", var.j))
    }

    if(ncol(recall.covariate.matrix) > 0) {

      cross <- t(recall.covariate.matrix) %*% recall.covariate.matrix
      eigvals <- eigen(cross, symmetric=TRUE, only.values=TRUE)$values
      if(any(eigvals < sqrt(.Machine$double.eps))) {

        logger <- update_logger(logger, 3, "Design Matrices", paste0("Linearly dependent columns in design matrix for ", var.j))
      }
    }
  }

  if(never.consumers.first.episodic) {

    never.consumer.covariate.matrix <- covariate.matrices$never.consumer

    if(ncol(never.consumer.covariate.matrix) == 0) {

      logger <- update_logger(logger, 3, "Design Matrices", paste0("No never-consumer covariates or intercept"))
    }

    if(ncol(never.consumer.covariate.matrix) > 0) {

      cross <- t(never.consumer.covariate.matrix) %*% never.consumer.covariate.matrix
      eigvals <- eigen(cross, symmetric=TRUE, only.values=TRUE)$values
      if(any(eigvals < sqrt(.Machine$double.eps))) {

        logger <- update_logger(logger, 3, "Design Matrices", paste0("Linearly dependent columns in never-consumer design matrix"))
      }
    }
  }

  return(logger)
}

#Log storage used by storage of U matrices
log_u_storage <- function(logger,
                          u.matrix.prior,
                          save.u.main,
                          save.all.u,
                          num.mcmc.iterations,
                          num.burn,
                          num.thin,
                          num.post) {

  #Main MCMC chain U matrices
  if(save.u.main) {

    if(save.all.u) {

      num.main <- num.mcmc.iterations
    } else {

      num.main <- (num.mcmc.iterations - num.burn) %/% num.thin
    }

    logger <- update_logger(logger, 1, "U Storage", paste0("Estimated memory for main chain MCMC U matrices: ", format(num.main*object.size(u.matrix.prior), units="auto", standard="SI")))
  }

  #Post-MCMC U matrices
  if(num.post > 0) {

    logger <- update_logger(logger, 1, "U Storage", paste0("Estimated memory for post-MCMC U matrices: ", format(num.post*object.size(u.matrix.prior), units="auto", standard="SI")))
  }

  return(logger)
}

#Checks input parameters to the MCMC function.
check_input <- function(logger,
                        num.mcmc.iterations,
                        num.burn,
                        num.thin,
                        num.post,
                        pre.mcmc.data,
                        id,
                        repeat.obs,
                        weight,
                        episodic.variables,
                        episodic.indicators,
                        episodic.amounts,
                        daily.variables,
                        daily.amounts,
                        never.consumer.variable,
                        never.consumer.indicator,
                        never.consumer.amount,
                        default.covariates,
                        episodic.indicator.covariates,
                        episodic.amount.covariates,
                        daily.amount.covariates,
                        individual.covariates,
                        never.consumer.covariates,
                        sigma.u.prior) {

  logger <- check_iterations(logger=logger,
                             num.mcmc.iterations=num.mcmc.iterations,
                             num.burn=num.burn,
                             num.thin=num.thin,
                             num.post=num.post)

  logger <- check_data(logger=logger,
                       pre.mcmc.data=pre.mcmc.data,
                       id=id,
                       repeat.obs=repeat.obs,
                       weight=weight)

  logger <- check_variable_lists(logger=logger,
                                 pre.mcmc.data=pre.mcmc.data,
                                 episodic.variables=episodic.variables,
                                 episodic.indicators=episodic.indicators,
                                 episodic.amounts=episodic.amounts,
                                 daily.variables=daily.variables,
                                 daily.amounts=daily.amounts,
                                 never.consumer.variable=never.consumer.variable,
                                 never.consumer.indicator=never.consumer.indicator,
                                 never.consumer.amount=never.consumer.amount)

  logger <- check_covariate_lists(logger=logger,
                                  pre.mcmc.data=pre.mcmc.data,
                                  default.covariates=default.covariates,
                                  episodic.indicator.covariates=episodic.indicator.covariates,
                                  episodic.amount.covariates=episodic.amount.covariates,
                                  daily.amount.covariates=daily.amount.covariates,
                                  individual.covariates=individual.covariates,
                                  never.consumer.covariates=never.consumer.covariates,
                                  episodic.variables=episodic.variables,
                                  episodic.indicators=episodic.indicators,
                                  episodic.amounts=episodic.amounts,
                                  daily.variables=daily.variables,
                                  daily.amounts=daily.amounts,
                                  never.consumer.variable=never.consumer.variable,
                                  never.consumer.indicator=never.consumer.indicator,
                                  never.consumer.amount=never.consumer.amount)

  logger <- check_sigma_u_prior(logger=logger,
                                sigma.u.prior=sigma.u.prior,
                                pre.mcmc.data=pre.mcmc.data,
                                episodic.variables=episodic.variables,
                                episodic.indicators=episodic.indicators,
                                episodic.amounts=episodic.amounts,
                                daily.variables=daily.variables,
                                daily.amounts=daily.amounts,
                                never.consumer.variable=never.consumer.variable,
                                never.consumer.indicator=never.consumer.indicator,
                                never.consumer.amount=never.consumer.amount)

  if(any(logger$record$error.code == 3)) {

    stop("Error(s) in input parameters")
  }

  return(logger)
}

#Checks MCMC initialized parameters before starting algorithm.
check_initialized <- function(logger,
                              covariate.matrices,
                              mcmc.priors,
                              save.u.main,
                              save.all.u,
                              num.mcmc.iterations,
                              num.burn,
                              num.thin,
                              num.post,
                              never.consumers.first.episodic) {

  logger <- check_design_matrices(logger=logger,
                                  covariate.matrices=covariate.matrices,
                                  never.consumers.first.episodic=never.consumers.first.episodic)

  logger <- log_u_storage(logger=logger,
                          u.matrix.prior=mcmc.priors$u.matrix.prior,
                          save.u.main=save.u.main,
                          save.all.u=save.all.u,
                          num.mcmc.iterations=num.mcmc.iterations,
                          num.burn=num.burn,
                          num.thin=num.thin,
                          num.post=num.post)

  if(any(logger$record$error.code == 3)) {

    stop("Error(s) in MCMC priors.")
  }

  return(logger)
}

