#Produces simulated Monte Carlo datasets using MCMC multivar parameters.
distrib_main_loop <- function(xbeta,
                              u.standard.deviation,
                              sigma.e.mean,
                              mcmc.u.matrices,
                              use.mcmc.u.matrices,
                              backtransformation.data,
                              records,
                              subjects,
                              subjects.mcmc,
                              episodic.variables,
                              episodic.indicators,
                              episodic.amounts,
                              daily.variables,
                              daily.amounts,
                              num.records,
                              num.subjects,
                              num.episodic,
                              num.daily,
                              num.replicates,
                              nuisance.weighting,
                              dietary.supplement.data,
                              distrib.population,
                              id,
                              weight,
                              additional.output,
                              never.consumers.first.episodic,
                              consumer.probabilities) {

  #unique subjects in distrib population
  distrib.subjects <- distrib.population[!duplicated(distrib.population[,id]),]

  #If MCMC U matrices are used, determine which rows of U matrices come from MCMC U matrices and which must be simulated
  mcmc.u.index <- NULL
  mcmc.distrib.match <- NULL
  sim.u.index <- NULL
  if(use.mcmc.u.matrices) {

    if(any(subjects %in% subjects.mcmc)) {

      mcmc.u.index <- which(subjects %in% subjects.mcmc)
      mcmc.distrib.match <- match(subjects[mcmc.u.index], subjects.mcmc)
    }

    if(any(!(subjects %in% subjects.mcmc))) {

      sim.u.index <- which(!(subjects %in% subjects.mcmc))
    }
  }

  #match records in distrib to subjects
  subject.record.match <- match(records, subjects)

  #main loop of replicates
  distrib.output <- vector(mode="list", length=num.replicates)
  mcmc.u.matrix <- NULL
  for(replicate in seq_len(num.replicates)) {

    #generate U matrix from MCMC output and simulated data (if needed)
    if(use.mcmc.u.matrices) {

      mcmc.u.matrix <- as.matrix(mcmc.u.matrices[,,replicate])
    }

    u.matrix <- generate_u_matrix(use.mcmc.u.matrices=use.mcmc.u.matrices,
                                  mcmc.u.matrix=mcmc.u.matrix,
                                  u.standard.deviation=u.standard.deviation,
                                  mcmc.u.index=mcmc.u.index,
                                  mcmc.distrib.match=mcmc.distrib.match,
                                  sim.u.index=sim.u.index,
                                  num.subjects=num.subjects,
                                  num.episodic=num.episodic,
                                  num.daily=num.daily)

    #calculate XBeta + U
    xbeta.u <- xbeta + u.matrix[subject.record.match,]

    #find subjects that are never-consumers of the first episodic food
    never.consumers <- find_never_consumers(never.consumers.first.episodic=never.consumers.first.episodic,
                                            consumer.probabilities=consumer.probabilities,
                                            subject.record.match=subject.record.match,
                                            num.subjects=num.subjects)

    #calculate consumption probabilities of each episodic food for each subject
    consumption.probability <- calculate_consumption_probability(xbeta.u=xbeta.u,
                                                                 episodic.variables=episodic.variables,
                                                                 num.records=num.records,
                                                                 num.episodic=num.episodic,
                                                                 never.consumers.first.episodic=never.consumers.first.episodic,
                                                                 never.consumers=never.consumers)

    #calculate backtransformed amount of each episodic food for each subject
    backtransformed.amount <- calculate_backtransformed_amount(xbeta.u=xbeta.u,
                                                               sigma.e.mean=sigma.e.mean,
                                                               backtransformation.data=backtransformation.data,
                                                               episodic.variables=episodic.variables,
                                                               daily.variables=daily.variables,
                                                               num.records=num.records,
                                                               num.episodic=num.episodic,
                                                               num.daily=num.daily,
                                                               never.consumers.first.episodic=never.consumers.first.episodic,
                                                               never.consumers=never.consumers)

    #calculate usual intake (t values)
    usual.intake <- calculate_usual_intake(consumption.probability=consumption.probability,
                                           backtransformed.amount=backtransformed.amount,
                                           episodic.variables=episodic.variables,
                                           daily.variables=daily.variables,
                                           num.records=num.records,
                                           num.episodic=num.episodic,
                                           num.daily=num.daily,
                                           never.consumers.first.episodic=never.consumers.first.episodic,
                                           never.consumers=never.consumers)

    #calculate supplemented intake
    supplemented.intake <- calculate_supplemented_intake(usual.intake=usual.intake,
                                                         dietary.supplement.data=dietary.supplement.data,
                                                         episodic.variables=episodic.variables,
                                                         daily.variables=daily.variables,
                                                         num.records=num.records)


    #combine intakes, probabilities, and amounts and sum over nuisance variable levels
    distrib.data <- cbind(usual.intake, supplemented.intake, consumption.probability, backtransformed.amount[,seq_len(num.episodic),drop=FALSE])
    distrib.replicate <- aggregate(distrib.data*nuisance.weighting, by=distrib.population[,id,drop=FALSE], sum)

    #merge in subject weights and additional variables
    distrib.replicate <- merge(distrib.subjects[,c(id, weight, additional.output),drop=FALSE],
                               distrib.replicate,
                               by=id)

    #save distrib replicate
    distrib.output[[replicate]] <- data.frame(replicate=replicate, distrib.replicate)
  }
  combined.distrib.output <- do.call(rbind, distrib.output)

  return(combined.distrib.output)
}
