#Initializes parameters for multivar distrib main loop.
initialize_distrib <- function(distrib.population,
                               id,
                               nuisance.weight,
                               mcmc.covariates,
                               mcmc.intercepts,
                               beta,
                               sigma.u,
                               sigma.e,
                               use.mcmc.u.matrices,
                               dietary.supplements,
                               num.simulated.u,
                               num.episodic,
                               num.daily,
                               num.mcmc.iterations,
                               num.burn,
                               num.thin,
                               num.post,
                               distrib.seed,
                               never.consumers.first.episodic,
                               never.consumer.covariates,
                               never.consumer.intercept,
                               alpha1) {

  #1. Set seed
  if(is.null(distrib.seed)) {

    distrib.seed <- as.integer(runif(1, min=-0.5, max=0.5)*2*10^7)
  }
  set.seed(distrib.seed)

  #2. Calculate population variables (sample sizes, number of covariate lists, number of variables)
  population.variables <- calculate_population_variables(distrib.population=distrib.population,
                                                         id=id,
                                                         nuisance.weight=nuisance.weight,
                                                         use.mcmc.u.matrices=use.mcmc.u.matrices,
                                                         dietary.supplements=dietary.supplements,
                                                         num.simulated.u=num.simulated.u,
                                                         num.post=num.post,
                                                         num.episodic=num.episodic,
                                                         num.daily=num.daily)

  #3. Extract covariate matrices
  covariate.matrices <- extract_covariate_matrices(distrib.population=distrib.population,
                                                   mcmc.covariates=mcmc.covariates,
                                                   mcmc.intercepts=mcmc.intercepts,
                                                   never.consumers.first.episodic=never.consumers.first.episodic,
                                                   never.consumer.covariates=never.consumer.covariates,
                                                   never.consumer.intercept=never.consumer.intercept)

  #4. Calculate means of MCMC multivar parameters
  mcmc.parameter.means <- calculate_mcmc_means(beta=beta,
                                               sigma.u=sigma.u,
                                               sigma.e=sigma.e,
                                               never.consumers.first.episodic=never.consumers.first.episodic,
                                               alpha1=alpha1,
                                               num.mcmc.iterations=num.mcmc.iterations,
                                               num.burn=num.burn,
                                               num.thin=num.thin)

  #5. Initialize distrib parameters
  distrib.parameters <- initialize_distrib_parameters(covariate.matrices=covariate.matrices,
                                                      beta.mean=mcmc.parameter.means$beta,
                                                      sigma.u.mean=mcmc.parameter.means$sigma.u,
                                                      sigma.e.mean=mcmc.parameter.means$sigma.e,
                                                      num.records=population.variables$num.records,
                                                      num.episodic=num.episodic,
                                                      num.daily=num.daily,
                                                      never.consumers.first.episodic=never.consumers.first.episodic,
                                                      g.matrix=covariate.matrices$never.consumer,
                                                      alpha1.mean=mcmc.parameter.means$alpha1)

  #6. Output distrib parameters
  output.parameters <- c(distrib.parameters,
                         population.variables)
  output.parameters$distrib.seed <- distrib.seed
  return(output.parameters)
}



#Calculate population variables (sample sizes, nuisance variable levels, number of variables).
calculate_population_variables <- function(distrib.population,
                                           id,
                                           nuisance.weight,
                                           use.mcmc.u.matrices,
                                           dietary.supplements,
                                           num.simulated.u,
                                           num.post,
                                           num.episodic,
                                           num.daily) {

  distrib.population <- as.data.frame(distrib.population)

  #Extract subjects
  records <- distrib.population[,id,drop=TRUE]
  subjects <- unique(records)

  #calculate total number of records and variables
  num.records <- length(records)
  num.subjects <- length(subjects)

  #extract nuisance variable level weighting and normalize the sum of weights for each subject to 1
  if(!is.null(nuisance.weight)) {

    nuisance.weighting <- distrib.population[,nuisance.weight,drop=TRUE]
  } else {

    nuisance.weighting <- rep(1, num.records)
  }
  nuisance.weighting <- ave(nuisance.weighting, records, FUN=function(weights) weights/sum(weights))

  #extract dietary supplement data
  if(!is.null(dietary.supplements)) {

    dietary.supplement.data <- matrix(nrow=nrow(distrib.population), ncol=length(dietary.supplements))
    for(i in seq_along(dietary.supplements)) {

      dietary.supplement.data[,i] <- distrib.population[,dietary.supplements[[i]]]
    }
    colnames(dietary.supplement.data) <- names(dietary.supplements)
  } else {

    dietary.supplement.data <- NULL
  }

  #Calculate number of replicates
  if(use.mcmc.u.matrices) {

    num.replicates <- num.post
  } else {

    num.replicates <- num.simulated.u
  }

  #return population variables in list
  population.variables <- list(records=records,
                               subjects=subjects,
                               nuisance.weighting=nuisance.weighting,
                               dietary.supplement.data=dietary.supplement.data,
                               num.records=num.records,
                               num.subjects=num.subjects,
                               num.replicates=num.replicates)
  return(population.variables)
}



#Extract covariate data for each variable from input dataset.
extract_covariate_matrices <- function(distrib.population,
                                       mcmc.covariates,
                                       mcmc.intercepts,
                                       never.consumers.first.episodic,
                                       never.consumer.covariates,
                                       never.consumer.intercept) {

  covariate.matrices <- vector(mode="list", length=2)
  names(covariate.matrices) <- c("variables", "never.consumer")

  #Extract recall covariate matrices
  covariate.matrices$variables <- vector(mode="list", length=length(mcmc.covariates))
  names(covariate.matrices$variables) <- names(mcmc.covariates)
  for(var.j in names(mcmc.covariates)) {

    covariate.matrices$variables[[var.j]] <- as.matrix(distrib.population[,mcmc.covariates[[var.j]],drop=FALSE])
    if(mcmc.intercepts[[var.j]]) {

      covariate.matrices$variables[[var.j]] <- cbind(intercept=1, covariate.matrices$variables[[var.j]])
    }
  }

  #extract never-consumer covariate matrix
  if(never.consumers.first.episodic) {

    covariate.matrices$never.consumer <- as.matrix(distrib.population[,never.consumer.covariates,drop=FALSE])
    if(never.consumer.intercept) {

      covariate.matrices$never.consumer <- cbind(intercept=1, covariate.matrices$never.consumer)
    }
  }

  return(covariate.matrices)
}



#Calculates posterior means of beta, sigma-u, sigma-e, and alpha1.
calculate_mcmc_means <- function(beta,
                                 sigma.u,
                                 sigma.e,
                                 never.consumers.first.episodic,
                                 alpha1,
                                 num.mcmc.iterations,
                                 num.burn,
                                 num.thin) {

  #iterations to use in means
  thinned.iterations <- seq(num.burn+1, num.mcmc.iterations, num.thin)

  #calculate means
  beta.mean <- vector(mode="list", length=length(beta))
  for(var.j in seq_along(beta)) {

    beta.mean[[var.j]] <- rowMeans(beta[[var.j]][,thinned.iterations,drop=FALSE])
  }
  names(beta.mean) <- names(beta)

  sigma.e.mean <- rowMeans(sigma.e[,,thinned.iterations,drop=FALSE], dims=2)

  sigma.u.mean <- rowMeans(sigma.u[,,thinned.iterations,drop=FALSE], dims=2)

  #if never-consumers are allowed, calculate alpha1 mean
  if(never.consumers.first.episodic) {

    alpha1.mean <- rowMeans(alpha1[,thinned.iterations,drop=FALSE])
  } else {

    alpha1.mean <- NULL
  }

  #output list of MCMC means
  mcmc.means <- list(beta=beta.mean,
                     sigma.u=sigma.u.mean,
                     sigma.e=sigma.e.mean,
                     alpha1=alpha1.mean)
  return(mcmc.means)
}



#Calculates XBeta, U standard deviation (square root of U), sample sizes, and consumer probabilities.
initialize_distrib_parameters <- function(covariate.matrices,
                                          beta.mean,
                                          sigma.u.mean,
                                          sigma.e.mean,
                                          num.records,
                                          num.episodic,
                                          num.daily,
                                          never.consumers.first.episodic,
                                          g.matrix,
                                          alpha1.mean) {

  #calculate XBeta
  num.variables <- 2*num.episodic + num.daily
  xbeta <- matrix(nrow=num.records, ncol=num.variables)
  for(var.j in seq_len(num.variables)) {

    xbeta[,var.j] <- covariate.matrices$variables[[var.j]] %*% beta.mean[[var.j]]
  }
  colnames(xbeta) <- names(beta.mean)

  #calculate the standard deviation of U
  sigma.u.eigen <- eigen(sigma.u.mean)
  u.standard.deviation <- sigma.u.eigen$vectors %*% diag(sqrt(sigma.u.eigen$values),nrow=num.variables, ncol=num.variables) %*% t(sigma.u.eigen$vectors)
  dimnames(u.standard.deviation) <- list(colnames(xbeta), colnames(xbeta))

  #if never-consumers are allowed, calculate consumer probabilities
  if(never.consumers.first.episodic) {

    galpha <- as.vector(g.matrix %*% alpha1.mean)
    consumer.probabilities <- pnorm(galpha)
  } else {

    consumer.probabilities <- NULL
  }

  #output list of distrib parameters
  distrib.parameters <- list(xbeta=xbeta,
                             u.standard.deviation=u.standard.deviation,
                             sigma.e.mean=sigma.e.mean,
                             consumer.probabilities=consumer.probabilities)
  return(distrib.parameters)
}
