/****************************************************************************/
/* The following files are required to run this example:                    */
/*   mixtran_macro_v2.21.sas                                                */
/*   distrib_macro_v2.2.sas                                                 */
/*   indiv_ims.sas7bdat                                                     */
/****************************************************************************/
/*                                                                          */
/****************************************************************************/
/* This example fits a univariate measurement model and estimates the       */
/* distribution of usual intake for a dietary component consumed            */
/* episodically.  For this analysis, the example data includes children     */
/* ages 1-8 from 2001-2004 public use NHANES, and the 24-hour dietary       */
/* recall is the main dietary instrument.  The dietary component is fruit.  */
/* In addition, recommended intakes (EAR and TUL) for several components    */
/* have been added to the data for each individual.  The example data       */
/* include repeated intake measurements from 24-hour dietary recalls.       */
/*                                                                          */
/* This analysis uses balanced repeated replication (BRR) variance          */
/* estimation, so the univariate measurement error model and distribution   */
/* of usual intake use the original weight and 32 BRR weights.  The         */
/* following replicfirst and repliclast macro variables allow specification */
/* of a range of replicates.  This program considers the original data set  */
/* (i.e. replicate 0).                                                      */
/****************************************************************************/

%let replicfirst = 0;
%let repliclast  = 0;



title1 "Fit Univariate Measurement Error Model Using MLE with 24-Hour Recall as Main Instrument";
title2 "Estimate Distribution of Usual Intake for a Dietary Component Consumed Episodically";
title3 "Using 2001-2004 NHANES Public Use Data for Children Ages 1-8";



***********************************************************************;
*** The path must be modified in the following lines to specify the ***;
*** location of the SAS macros, the SAS data files, and the SAS     ***;
*** library that will be used for output                            ***;
***********************************************************************;

%let home = /prj/dcp/statprog/meas.err/develop.public.resources.stat.meth;

*** Include the required macros ***;
%include "&home/include.files.macros/mixtran_macro_v2.21.sas";
%include "&home/include.files.macros/distrib_macro_v2.2.sas";

*** Input data libaray ***;
libname indata "&home/data";

*** Output data library ***;
libname outlib "&home/univar_surveillance_example2_mle_main24hr/outdata";



***********************************************************************************;
*** A macro is used to loop the original weight variable and the 32 BRR weights ***;
*** through calls to the MIXTRAN and DISTRIB macros.  To accomplish this        ***;
*** seamlessly the name of the original weight variable and the BRR weight      ***;
*** variables must have the same root and must end in 0-32, e.g. rndw0-rndw32,  ***;
*** where rndw0 is the original sample weight.                                  ***;
***********************************************************************************;

%let weight_var_orig = rndw;
%let weight_var      = rndw;



***********************************************************************************;
*** Global macro variable to test for successful execution of the MIXTRAN macro ***;
***********************************************************************************;

%global success;



***************;
*** Formats ***;
***************;

proc format;
  value nonzerof
                0 = '0'
    0.000001-1000 = '> 0';
  value agegf
      1 = '1-3'
      2 = '4-8'

   -255 = 'Overall';
  value sexfmt
    1 = 'Male'
    2 = 'Female';



***********************************************;
*** Read in the data and sort by subject id ***;
***********************************************;

proc sort data=indata.indiv_ims out=nhanes;
  by seqn;
run;



**********************************************************************************************;
*** Read in data, susbset to stratum of interest, and create dummies of analysis variables ***;
**********************************************************************************************;

data child;
  set nhanes(keep = seqn race agegrp age2 rndw rndw1-rndw32
                    day day2 f_total f_total_rec sex sex1 stratum weekend);
  by seqn ;

  if stratum = 1;  *** subset to children ages 1-8 ***;

  r_f_total = (f_total > 0);  *** total fruit consumption > 0 ***;

  race1 = (race = 1);
  race3 = (race = 3);
  seq2 = (day  = 2);  *** seq2 equals 1 if second day of recall and equals 0 otherwise ***;

  label r_f_total = "Fruit consumption greater than 0"
        race1     = "Race: Mexican/American (ref = White)"
        race3     = "Race: Non-Hispanic Black (ref = White)";

  ***********************************************************************;
  *** The original weight variable name does not end with zero (0),   ***;
  *** so it is renamed.  The new name has the same root name of the   ***;
  *** BRR weight variables and has 0 appended.                        ***;
  *** In this example the original weight variable name is rndw.      ***;
  *** The BRR weights are named rndw1-rndw32.                         ***;
  ***********************************************************************;

  rename &weight_var_orig = &weight_var.0;

run;



******************************************************************************************;
*** The macro fit_models_replicate_loop is used to call the MIXTRAN and DISTRIB macros ***;
******************************************************************************************;

%macro fit_models_replicate_loop;

  %do replicnum = &replicfirst %to &repliclast ;

    title4 "Replicate &replicnum";

    **********************************************;
    *** Use a macro variable to represent the  ***;
    *** replicate variables in subsequent code ***;
    **********************************************;

    %let replicvar=&weight_var&replicnum ;



    ********************************************************************************;
    ***       Call MIXTRAN to fit a nonlinear mixed model for total fruit        ***;
    ********************************************************************************;

    /*******************************************************************************/
    /*                                                                             */
    /*  THE MIXTRAN MACRO                                                          */
    /*                                                                             */
    /*******************************************************************************/
    /*                                                                             */
    /* The MIXTRAN macro is used for the analysis of episodically                  */
    /* consumed foods, foods consumed every day, and nutrients, and                */
    /* output from the MIXTRAN macro is used by the DISTRIB macro for              */
    /* estimation of the distribution of usual intake.                             */
    /*                                                                             */
    /* For episodically consumed foods, the MIXTRAN macro fits a two-              */
    /* part nonlinear mixed model where the first part considers the               */
    /* probability of consumption and the second part considers the                */
    /* consumption-day amount.  The model allows for covariates and                */
    /* includes a random effect in both parts and allows for correlation           */
    /* between the random effects (Tooze et al., 2006, Journal of the              */
    /* American Dietetic Association, 106, 1575-1587).                             */
    /*                                                                             */
    /* To fit this nonlinear mixed model with correlated random effects            */
    /* (i.e. the correlated model), starting values for the two parts of           */
    /* the model are obtained by first using the GENMOD procedure to fit           */
    /* a probability model and an amount model.  Then a nonlinear mixed            */
    /* model with uncorrelated random effects (i.e. the uncorrelated               */
    /* model) is fit using two calls to the NLMIXED procedure, and the             */
    /* parameter estimates from this model are used as starting values             */
    /* for the correlated model.                                                   */
    /*                                                                             */
    /* For foods consumed every day and nutrients, the MIXTRAN macro               */
    /* uses the GENMOD procedure to calculate starting values and uses             */
    /* the NLMIXED procedure to fit an amount-only model.                          */
    /*                                                                             */
    /* The syntax for calling the macro is:                                        */
    /*                                                                             */
    /* %MIXTRAN                                                                    */
    /* (data=, response=, foodtype=, subject=, repeat=,                            */
    /*  covars_prob=, covars_amt=, outlib=, modeltype=,                            */
    /*  lambda=, replicate_var=, seq=,                                             */
    /*  weekend=, vargroup=, numvargroups=,                                        */
    /*  start_val1=, start_val2=, start_val3=, vcontrol=,                          */
    /*  nloptions=, titles=, printlevel=,subgroup=)                                */
    /*                                                                             */
    /*  where                                                                      */
    /*                                                                             */
    /*   "data"         * Specifies the dataset to be used.                        */
    /*                                                                             */
    /*   "response"     * Specifies the 24 hour recall variable.                   */
    /*                                                                             */
    /*   "foodtype"     * Specifies a name for the analysis, used to               */
    /*                    identify the output data sets.  This value can           */
    /*                    be the same as the response variable.                    */
    /*                                                                             */
    /*   "subject"      * Specifies the variable that uniquely                     */
    /*                    identifies each subject.                                 */
    /*                                                                             */
    /*   "repeat"       * Specifies the variable that indexes repeated             */
    /*                    observations for each subject.                           */
    /*                                                                             */
    /*   "covars_prob"    Specifies a list of covariates for the first             */
    /*                    part of the model that models the probability            */
    /*                    of consumption.  Covariates must be separated            */
    /*                    by spaces.  Interactions must be in the order            */
    /*                    specified by PROC GENMOD.  If the model type             */
    /*                    is "amount" then covars_prob should be left as           */
    /*                    a null string.                                           */
    /*                                                                             */
    /*   "covars_amt"   * Specifies a list of covariates for the second            */
    /*                    part of the model that models the consumption-           */
    /*                    day amount.  Covariates must be separated by             */
    /*                    spaces.  Interactions must be in the order               */
    /*                    specified by PROC GENMOD.                                */
    /*                                                                             */
    /*   "outlib"       * Specifies a directory where output data sets             */
    /*                    are stored.  Outlib can not be null.                     */
    /*                                                                             */
    /*   "modeltype"    * Specifies the model.  The possible values are:           */
    /*                    "null string" = fit correlated model,                    */
    /*                    "corr"        = fit correlated model,                    */
    /*                    "nocorr"      = fit uncorrelated model,                  */
    /*                    "amount"      = fit amount-only model.                   */
    /*                                                                             */
    /*   "lambda"         Specifies a user supplied value for the                  */
    /*                    Box-Cox transformation parameter, lambda.  If            */
    /*                    a value is not supplied, the macro will                  */
    /*                    calculate a value for lambda.                            */
    /*                                                                             */
    /*   "replicate_var"  Specifies the variable to be used in the                 */
    /*                    replicate statement of PROC NLMIXED or the               */
    /*                    freq statement of PROC UNIVARIATE.  The                  */
    /*                    specified variable must be integer valued.               */
    /*                                                                             */
    /*   "seq"            Specifies one or more sequence indicator                 */
    /*                    variables to account for effects due to the              */
    /*                    sequence number of a subject's records.  This            */
    /*                    variable can NOT also appear in covars_prob              */
    /*                    or covars_amt.                                           */
    /*                                                                             */
    /*   "weekend"        Specifies the weekend (Fri.-Sun.) indicator              */
    /*                    variable to account for a weekend effect.  A             */
    /*                    value of 1 represents a Fri.-Sun. record, and            */
    /*                    a value of 0 represents a Mon.-Thurs. record.            */
    /*                    This variable can NOT also appear in                     */
    /*                    covars_prob or covars_amt.                               */
    /*                                                                             */
    /*   "vargroup"       Specifies a variable that groups observations            */
    /*                    to allow the model to incorporate a separate             */
    /*                    residual variance parameter for each of these            */
    /*                    groups of observations.  If the output from              */
    /*                    this macro is to be used in the DISTRIB macro,           */
    /*                    then only the weekend variable can be used.              */
    /*                                                                             */
    /*   "numvargroups"   Specifies the number of groups defined by the            */
    /*                    vargroup variable.  If the output from this              */
    /*                    macro is to be used in the DISTRIB macro and             */
    /*                    weekend is the "vargroup" variable, then the             */
    /*                    number of groups is 2.                                   */
    /*                                                                             */
    /*   "start_val1"     Starting values for probability model (nocorr).          */
    /*                    Use only when vcontrol is called and parameter           */
    /*                    estimates (i.e. _parmsf1_"foodtype") from a              */
    /*                    previous execution of an analogous model are desired.    */
    /*                    Specifies the starting values data set for the           */
    /*                    1st PROC NLMIXED (i.e. NLMIXED for probability           */
    /*                    model).                                                  */
    /*                                                                             */
    /*   "start_val2"     Starting values for the amount model.                    */
    /*                    Use only when vcontrol is called and parameter           */
    /*                    estimates (i.e. _parmsf2_"foodtype") from a              */
    /*                    previous execution of an analogous model are desired.    */
    /*                    Specifies the starting values data set for the           */
    /*                    2nd PROC NLMIXED (i.e. NLMIXED for amount                */
    /*                    model).                                                  */
    /*                                                                             */
    /*   "start_val3"     Starting values for correllated model (corr).            */
    /*                    Use only when vcontrol and parameter                     */
    /*                    estimates (i.e. _parmsf3_"foodtype") from a              */
    /*                    previous execution of an analogous model are desired.    */
    /*                    Specifies the starting values data set for the           */
    /*                    3rd PROC NLMIXED (i.e. NLMIXED for correlated            */
    /*                    model).                                                  */
    /*                                                                             */
    /*   "vcontrol"       Use only when starting values from a previous            */
    /*                    execution of the same model are also used.               */
    /*                    Specifies a 1 to 6 character name to                     */
    /*                    differentiate output data sets for runs using            */
    /*                    the same food.  See the parameters start_val1,           */
    /*                    start_val2, and start_val3.  The default is              */
    /*                    null.                                                    */
    /*                                                                             */
    /*   "nloptions"      Specifies a list of options to be added to all           */
    /*                    calls to PROC NLMIXED, for example:                      */
    /*                       nloptions=qpoints=1 gconv=1e-12 itdetails.            */
    /*                                                                             */
    /*   "titles"         Specifies the number of title lines (0-4) to             */
    /*                    be reserved for the user's titles.  Up to 4              */
    /*                    title lines may be reserved for the user's               */
    /*                    titles.  The remaining title lines are used by           */
    /*                    the macro.  The default value is 0.                      */
    /*                                                                             */
    /*   "printlevel"     Specifies 1, 2, or 3 to control the amount of            */
    /*                    information printed in the list file.                    */
    /*                    Printlevel=1 prints only the summary reports.            */
    /*                    Printlevel=2 prints summary reports and output           */
    /*                    from the NLMIXED procedures.  Printlevel=2 is            */
    /*                    the default value.  Printlevel=3 prints                  */
    /*                    summary reports amd output from all of the               */
    /*                    statistical prodecures.                                  */
    /*                                                                             */
    /*   "subgroup"       Specifies one categorical variable used for              */
    /*                    the calculation of a separate usual intake               */
    /*                    distribution for each subgroup.  This variable           */
    /*                    can be created from a combination of other               */
    /*                    variables (e.g. age and sex) but all variables           */
    /*                    used to define the subgroup variable must also           */
    /*                    be among the covariates in the model.  The               */
    /*                    subgroup variable is used in the DISTRIB                 */
    /*                    macro; however it can optionally be included             */
    /*                    in the call to the MIXTRAN macro to achieve              */
    /*                    backward compatibility with version 1.1                  */
    /*                    Calling a subgroup variable in MIXTRAN does not limit    */
    /*                    users to only the named subgroup in DISTRIB.             */
    /*                    A different subgroup variable can be called in DISTRIB   */
    /*                    but see documentation for DISTRIB on how to do this.     */
    /*                    The subgroup parameter can now be called in DISTRIB.     */
    /*                                                                             */
    /*                                                                             */
    /* Note:  * Parameters marked with an asterisk are mandatory, so a             */
    /*          value must be supplied in the macro call.                          */
    /*                                                                             */
    /* Caution:  variable name "YN" is reserved for this macro.                    */
    /*                                                                             */
    /* Caution:  data set names "data" and "data0" and "misc_info" are             */
    /*           reserved for this macro.                                          */
    /*                                                                             */
    /*******************************************************************************/

    %mixtran(data          = child,
             response      = f_total,
             modeltype     = corr,
             foodtype      = f_total_child&replicnum,
             subject       = seqn,
             repeat        = day,
             outlib        = outlib,
             covars_prob   = sex1 age2 race1 race3,
             covars_amt    = sex1 age2 race1 race3,
             seq           = day2,
             lambda        = ,
             replicate_var = &replicvar,
             weekend       = weekend,
             vargroup      = ,
             numvargroups  = ,
             vcontrol      = ,
             start_val1    = ,
             start_val2    = ,
             start_val3    = ,
             nloptions     = qmax=61,
             titles        = 4,
             printlevel    = 2
             );

    *****************************************************************************;
    **** Note: The parameters with null values can be omitted from this call. ***;
    ****       They are included in the example for documentation.            ***;
    *****************************************************************************;


    ********************************************************;
    *** If the MIXTRAN macro executed successfully then  ***;
    *** continue with this iteration of the example,     ***;
    *** otherwise stop this iteration and begin the next ***;
    ********************************************************;

    %if &success = 1 %then %do;



      ********************************************************************************;
      ***        Call DISTRIB to calculate the estimated usual intake and          ***;
      ***               percentile distribution for total fruit                    ***;
      ********************************************************************************;

      /*******************************************************************************/
      /*  THE DISTRIB MACRO                                                          */
      /*                                                                             */
      /*******************************************************************************/
      /*                                                                             */
      /*                                                                             */
      /* The DISTRIB macro uses results from the MIXTRAN macro and estimates the     */
      /* distribution of usual intake for episodically consumed foods, foods         */
      /* consumed every day, and nutrients (Tooze et al., 2006, Journal of the       */
      /* American Dietetic Association, 106, 1575-1587).  The data can then be used  */
      /* to calculate percentiles and, optionally, the percent meeting the           */
      /* recommended daily intake for a population.                                  */
      /*                                                                             */
      /* The DISTRIB macro contains two main functions.                              */
      /*                                                                             */
      /* First, the DISTRIB macro reads data sets of parameter estimates and         */
      /* predicted values output by the MIXTRAN macro.  Monte Carlo simulation of    */
      /* the random effect(s) is used to estimate the distribution of usual intake.  */
      /* This data set can be saved.                                                 */
      /*                                                                             */
      /* Second, once the data containing the estimated usual intake are available,  */
      /* percentiles and cutpoints can be calculated.  The addition of a sub group   */
      /* variable is accommodated, so that statistics can be calculated by subgroup  */
      /* and for the overall data set. Optionally the percent who meet recommended   */
      /* daily intake values can be calculated.                                      */
      /*                                                                             */
      /* To accomplish this and allow flexibility,the DISTRIB macro contains two     */
      /* sub-macros and some general code to set up and call the macros as           */
      /* requested.                                                                  */
      /*                                                                             */
      /* the macro MC uses monte carlo simulation of the random effect(s) to         */
      /* estimate the distribution of usual intake.  The output data set can be      */
      /* saved for future use.                                                       */
      /*                                                                             */
      /* The macro PC reads in the usual intake values calculated in the macro MC,   */
      /* normalises the weights, calculates the percentiles of usual intake,         */
      /* cutpoints if requested, and optionally the percent meeting recommended      */
      /* intake.  A single subgroup variable can be accommodated in the macro PC.    */
      /* The resulting data set can be saved for future use.                         */
      /*                                                                             */
      ********************************************************************************/
      /*                                                                             */
      /*                                                                             */
      /* The syntax for calling the DISTRIB macro is:                                */
      /*                                                                             */
      /*       %DISTRIB(call_type     = ,                                            */
      /*                seed          = ,                                            */
      /*                nsim_mc       = ,                                            */
      /*                modeltype     = ,                                            */
      /*                pred          = ,                                            */
      /*                param=,outlib = ,                                            */
      /*                cutpoints     = ,                                            */
      /*                ncutpnt       = ,                                            */
      /*                byvar         = ,                                            */
      /*                subgroup      = ,                                            */
      /*                add_da        = ,                                            */
      /*                subject       = ,                                            */
      /*                titles        = ,                                            */
      /*                food          = ,                                            */
      /*                mcsimda       = ,                                            */
      /*                recamt        = ,                                            */
      /*                recamt_co     = ,                                            */
      /*                recamt_hi     = ,                                            */
      /*                wkend_prop    = ,                                            */
      /*                wkend_mc      =                                             */
      /*                );                                                           */
      /*                                                                             */
      /* where:                                                                      */
      /*                                                                             */
      /*  "call_type" * Specifies which parts of the DISTRIB macro should be         */
      /*                invoked. (FULL, MC, PC). FULL is the default.                */
      /*                A null string implies FULL.                                  */
      /*                FULL = invoke both the calculation of the estimated intake   */
      /*                  of the estimated intake amount (using Monte Carlo          */
      /*                  simulation) and of the percentiles (and optionally the     */
      /*                  percent not meeting the recommended amount of intake).     */
      /*                MC = restrict the macro to calculating the intake amount     */
      /*                  (using Monte Carlo simulation).                            */
      /*                PC = use the intake estimates - calculated in the MC macro   */
      /*                  in DISTRIB to calculate percentiles and, optionally, the   */
      /*                  percent meeting the recommended amount ofintake, and/or    */
      /*                  cutpoints.                                                 */
      /*                                                                             */
      /*  "seed"      * Specifies the seed for the random number generator used for  */
      /*                the Monte Carlo simulation of the random effects u1 and u2.  */
      /*                Required if the call_type is FULL or MC.                     */
      /*                Not used if call_type is PC.                                 */
      /*                                                                             */
      /*  "nsim_mc"   * Specifies the number of repetitions to be used in the Monte  */
      /*                Carlo simulation.  For each subject, one record will be      */
      /*                output for each repetition.                                  */
      /*                Required if the call_type is FULL or MC.                     */
      /*                Not used if call_type is PC.                                 */
      /*                                                                             */
      /*  "modeltype" * Specifies the model that was used by the MIXTRAN macro to    */
      /*                prepare the data for the DISTRIB macro.                      */
      /*                The value must be the same as the model declared for the     */
      /*                MIXTRAN macro. The default is correlated.                    */
      /*                The possible values are:                                     */
      /*                  null string  = fit correlated model,                       */
      /*                  corr         = fit correlated model,                       */
      /*                  nocorr       = fit uncorrelated model,                     */
      /*                  amount       = fit amount-only model.                      */
      /*                                                                             */
      /*  "pred"      * Specifies the name of the data set containing predicted      */
      /*                values calculated in the MIXTRAN macro.                      */
      /*                Required.                                                    */
      /*                                                                             */
      /*  "param"     * Specifies the name of the data set containing the parameter  */
      /*                estimates calculated in the MIXTRAN macro.                   */
      /*                Required if the call_type is FULL or MC.                     */
      /*                Not used if call_type is PC.                                 */
      /*                                                                             */
      /*  "outlib"    * Specifies the library reference to which the output data set */
      /*                of distributions will be written.                            */
      /*                                                                             */
      /*  "cutpoints"   Specifies a list of cutoff points separated by a space.      */
      /*                Not used if call_type is MC.                                 */
      /*                                                                             */
      /*  "ncutpnt"     Specifies the number of cutoff points.  If cutoff points are */
      /*                given, ncutpnt must also be given.                           */
      /*                Not used if call_type is MC.                                 */
      /*                                                                             */
      /*  "byvar"       Specifies a list of by-variables that are in the data sets   */
      /*                "pred" and "param" to indicate that the MIXTRAN model was    */
      /*                fit separately for each by-group. The estimates used in the  */
      /*                calculation will differ based on the by group, however the   */
      /*                DISTRIB macro will ultimately produce estimates of the       */
      /*                entire population (not distributions within each by group).  */
      /*                To obtain distributions for subpopulations, use the          */
      /*                "subgroup" parameter.                                        */
      /*                                                                             */
      /*  "subgroup"    Specifies one categorical variable used for the calculation  */
      /*                of a separate usual intake distribution for each subgroup.   */
      /*                The distribution of usual intake will also be calculated for */
      /*                the overall data set. Requires that the paramater add_da be  */
      /*                supplied.                                                    */
      /*                Not used if call_type is MC.                                 */
      /*                                                                             */
      /*  "add_da"      The name of the data set containing the subgroup variable by */
      /*                which the percentiles are to be calculated, and/or the       */
      /*                recommended amount variable.                                 */
      /*                This data set must include:                                  */
      /*                  the ID variable declared in the parameter SUBJECT,         */
      /*                  and one or both of the following variables:                */
      /*                    the variable named in the parameter SUBGROUP             */
      /*                    the variable(s) named in the parameter(s) recamt and/or  */
      /*                    recam*t_hi.                                              */
      /*                This parameter is required if either of the parameters       */
      /*                subgroup or recamt are called.                               */
      /*                Not used if call_type is MC.                                 */
      /*                                                                             */
      /*  "subject" *   Specifies the variable that uniquely identifies each         */
      /*                subject. (The ID.)                                           */
      /*                                                                             */
      /*                                                                             */
      /*  "titles"      Specifies the number of title lines to be reserved for the   */
      /*                user's titles.                                               */
      /*                The default value is 0.                                      */
      /*                                                                             */
      /*  "food"     *  Specifies a name for the analysis, used to name the output   */
      /*                data set.                                                    */
      /*                                                                             */
      /*  "mcsimda"  *  Specifies the name of the data set containing the intake     */
      /*                amount derived from the Monte Carlo simulations. To read or  */
      /*                write the data file from disk include a libname.  Note: due  */
      /*                to simulations (see parameter nsim_mc) the data set can grow */
      /*                quite large.                                                 */
      /*                Default value is work.mcsim in which case the data set is    */
      /*                not saved for later use.                                     */
      /*                Required if call_type is PC.                                 */
      /*                                                                             */
      /*  "recamt"     The name of the variable containing the cut off level for the */
      /*               recommended amount of consumption for this food.              */
      /*               If the value of the "recamt_co" parameter is R then this      */
      /*               variable is used as the lower limit of the range.             */
      /*               If this parameter is used the name of the data set containing */
      /*               this variable must be supplied via the parameter "add_da".    */
      /*               Not used if the call_type is MC.                              */
      /*                                                                             */
      /* "recamt_co"   The Comparison Operator between individual intake and the     */
      /*               recommended amount described in the "recamt" parameter.       */
      /*               Options are:                                                  */
      /*                 LT - less than the "recamt" value,                          */
      /*                 LE - less than or equal to the "recamt" value,              */
      /*                 GE - greater than or equal to the "recamt" value,           */
      /*                 GT - greater than the "recamt" value,                       */
      /*                 R  - a range of values between two proportions, inclusive.  */
      /*               If the R option is used then the lower value will be the      */
      /*               value of the  variable in the "recamt" paramater, and the     */
      /*               upper value must be provided via the paramter "recamt_hi".    */
      /*               The "recamt_co" parameter is required if the "recamt"         */
      /*               parameter is supplied.                                        */
      /*               Not used if the call type is MC.                              */
      /*                                                                             */
      /* "recamt_hi"   The name of the variable containing the upper limit of a      */
      /*               range of inclusive values used to compare intake to a         */
      /*               recommended amount. This parameter is required if the value   */
      /*               of the "recamt_co" parameter is R.                            */
      /*               Not used if the call type is MC.                              */
      /*                                                                             */
      /* wkend_prop    A value between 0 and 1 (not inclusive).                      */
      /*               This parameter specifies the proportional weight for the      */
      /*               weekend days if "weekend" was used in the MIXTRAN macro.      */
      /*               Either a fraction or decimal number is acceptable.            */
      /*               The remaining (e.g. weekday) proportion is calculated within  */
      /*               the macro. The default weights for weekdays and weekend days  */
      /*               are 4/7 and 3/7 respectively.                                 */
      /*               Note: it is possible to use the "weekend" and "wkend_prop"    */
      /*               parameters with a binary variable other than a weekend        */
      /*               indicator.                                                    */
      /*               Not used if the call_type is PC.                              */
      /*                                                                             */
      /*  wkend_mc     YES or a null string.                                         */
      /*               YES specifies that separate estimates of consumption day or   */
      /*               total amounts by weekend status be stored in the _mc_sim      */
      /*               data files. Ohtrewise these interim variables will not be     */
      /*               kept.                                                         */
      /*               This paramter is only used in a weekend run.                  */
      /*               Not used if the call_type is PC.                              */
      /*                                                                             */
      /* Note:  * Parameters marked with an asterisk are mandatory, a value must be  */
      /*          supplied in the macro call.                                        */
      /*                                                                             */
      /*******************************************************************************/;

      %let loopseed_distrib = %eval(89009890 + &replicnum * 10000);

      %distrib(call_type = Full,
               modeltype = corr,
               outlib    = outlib,
               subject   = seqn,
               food      = f_total_child,
               mcsimda   = outlib._mcsim_f_total_child_rndw&replicnum,
               pred      = outlib._pred_f_total_child&replicnum,
               param     = outlib._param_f_total_child&replicnum,
               seed      = &loopseed_distrib,
               nsim_mc   = 100,
               cutpoints = 0.7 1.1 1.6,
               ncutpnt   = 3,
               subgroup  = agegrp,
               add_da    = child,
               recamt    = f_total_rec,
               recamt_co = GE,
               titles    = 4
               );

    ******************************************************************;
    *** End of the processing of the DISTRIB macro for iterations  ***;
    *** that were successful in the MIXTRAN macro                  ***;
    ******************************************************************;

    %end;


    %else %do;

      ************************************************;
      *** Exit the loop if MIXTRAN is unsuccessful ***;
      ************************************************;

      %put MIXTRAN was not successful for replicate data set &replicnum - DISTRIB macro was not executed;

    %end;


  %end;   *** end of the replicate loop ***;


%mend fit_models_replicate_loop;


************************************************;
*** Call the fit_models_replicate_loop macro ***;
************************************************;

%fit_models_replicate_loop;
