options mergenoby=nowarn;

/***************************************************************************/
/* The following files are required to run this example:                   */
/*   mixtran_macro_v2.21.sas                                               */
/*   distrib_macro_v2.2.sas                                                */
/*   boxcox_survey.macro.v1.2.sas                                          */
/*   indivint_macro_v2.3.sas                                               */
/*   data.example.males12plus.xpt                                          */
/***************************************************************************/
/*                                                                         */
/***************************************************************************/
/* This example analysis uses regression calibration and fits a linear     */
/* regression model to assess the relationship between a dietary component */
/* and a health outcome.  For this analysis, the example data include      */
/* males, ages 12+ from NHANES 2003-2004, and the 24-hour dietary recall   */
/* is the main dietary instrument.  The dietary component is fish measured */
/* in ounces, and the health outcome is serum mercury level measured in    */
/* micrograms per liter.  The mercury level is fit on a natural log scale. */
/* The example data include repeated intake measurements from 24-hour      */
/* dietary recalls.                                                        */
/*                                                                         */
/* This analysis uses balanced repeated replication (BRR) variance         */
/* estimation, so the univariate measurement error model and linear        */
/* regression model are fit using the original weight and 16 BRR weights.  */
/* The following replicfirst and repliclast macro variables allow          */
/* specification of a range of replicates.  This program considers the     */
/* original data set (i.e. replicate 0).                                   */
/***************************************************************************/

%let replicfirst = 0;
%let repliclast  = 0;



title1 "Fit Univariate Measurement Error Model Using MLE with 24-Hour Recall as Main Instrument";
title2 "Predict Intake and Perform Regression Calibration";
title3 "Assess Relationship between a Dietary Component and a Health Outcome";



*****************************************************************************;
*** Specify the appropriate paths for the following files and SAS library ***;
*****************************************************************************;

%let home = /prj/dcp/statprog/meas.err/develop.public.resources.stat.meth;

*** Include the required macros ***;
%include "&home/include.files.macros/mixtran_macro_v2.21.sas";
%include "&home/include.files.macros/distrib_macro_v2.2.sas";
%include "&home/include.files.macros/boxcox_survey.macro.v1.2.sas";
%include "&home/include.files.macros/indivint_macro_v2.3.sas";

*** Input data file ***;
filename tranfile "&home/data/data.example.males12plus.xpt";

*** Output SAS library ***;
libname outlib "&home/univar_epidemiology_example4_mle_main24hr/outlib";



************************************************************************************;
*** A macro is used to loop the original weight variable and the 16 BRR weights  ***;
*** through calls to the MIXTRAN and INDIVINT macros.  To accomplish this        ***;
*** seamlessly the name of the original weight variable and the BRR weight       ***;
*** variables must have the same root and must end in 0-16, e.g. rwthg0-rwthg16, ***;
*** where rwthg0 is the original weight. (In this example the original weight is ***;
*** rwthg.) The following macro variables are used to ensure proper naming.      ***;
************************************************************************************;

%let weight_var_orig = rwthg;
%let weight_var      = rwthg;



***********************************************************************************;
*** Global macro variable to test for successful execution of the MIXTRAN macro ***;
***********************************************************************************;

%global success;



*************************************************;
*** Comparison usual intake values: t0 and t1 ***;
*************************************************;

%let t0 = 0.1;
%let t1 = 1;



*********************************************;
*** Import the data file for this example ***;
*********************************************;

proc cimport data=examplemales12plus infile=tranfile;
run;



**********************************************************************;
*** Create separate data sets for the MIXTRAN and INDIVINT macros. ***;
*** The input data has one record per person.                      ***;
*** The MIXTRAN macro uses an input data set that includes one or  ***;
*** more observations for each person.                             ***;
*** The INDIVINT macro uses an input data set that includes one    ***;
*** observation for each person.                                   ***;
**********************************************************************;

data datamrec(keep = subjectid rwthg0-rwthg16 wave seqn numdays ridageyr race1 race3 dmdeduc1 dmdeduc3
                     ffqeqzero bcffq0mean R repeat)
     subj1rec(keep = subjectid rwthg0-rwthg16 wave seqn numdays ridageyr race1 race3 dmdeduc1 dmdeduc3
                     ffqeqzero bcffq0mean R1 R2 logtotmerc);
  set examplemales12plus;
  by subjectid;



  ***********************************************************************;
  *** The original weight variable name does not end with zero (0),   ***;
  *** so it is renamed.  The new name has the same root name of the   ***;
  *** BRR weight variables and has 0 appended.                        ***;
  *** In this example the original weight variable name is rwthg.     ***;
  *** The BRR weights are named rwthg1-rwthg16.                       ***;
  ***********************************************************************;

  rename &weight_var_orig = &weight_var.0;



  *****************************************************;
  *** Output record for 1st 24-hour recall.         ***;
  *** In this example R1 is the reported amount of  ***;
  *** fish consumed on the first 24-hour recall.    ***;
  *****************************************************;

  if R1 >= 0 then do;

    repeat = 1;

    R = R1;

    output datamrec;

  end;

  **************************************************;
  *** Output record for 2nd 24-hour recall.      ***;
  *** R2 is the reported amount of fish consumed ***;
  *** on the second 24-hour recall.              ***;
  **************************************************;

  if R2 >= 0 then do;

    repeat = 2;

    R = R2;

    output datamrec;

  end;

  ****************************************;
  *** Output 1 record for each subject ***;
  ****************************************;

  output subj1rec;

run;



************************************************************;
*** Calculate the smallest positive 24-hour recall value ***;
************************************************************;

proc univariate data=datamrec noprint;
  where R>0;
  var R;
  output out=outmin_amt min=min_amt;
run;



*********************************************************************************;
*** The macro fit_models_replicate_loop is used to call the MIXTRAN and       ***;
*** INDIVINT macros and fit the health outcome model using the original study ***;
*** data (i.e. replicate data set 0) and each of the replicate datasets (i.e. ***;
*** replicate datasets 1, 2, ..., 16) for balanced repeated replication (BRR) ***;
*** variance estimation                                                       ***;
*********************************************************************************;

%macro fit_models_replicate_loop;

  %do replicnum = &replicfirst %to &repliclast ;

    title4 "Replicate &replicnum";

    **********************************************;
    *** Use macro variable to represent the    ***;
    *** replicate variables in subsequent code ***;
    **********************************************;

    %let replicvar=&weight_var&replicnum ;



    /*******************************************************************************/
    /*******************************************************************************/
    /*                                                                             */
    /*  Description of the MIXTRAN Macro                                           */
    /*                                                                             */
    /*******************************************************************************/
    /*                                                                             */
    /* The MIXTRAN macro is used for the analysis of episodically                  */
    /* consumed foods, foods consumed every day, and nutrients, and                */
    /* output from the MIXTRAN macro is used by the DISTRIB macro for              */
    /* estimation of the distribution of usual intake.                             */
    /*                                                                             */
    /* For episodically consumed foods, the MIXTRAN macro fits a two-              */
    /* part nonlinear mixed model where the first part considers the               */
    /* probability of consumption and the second part considers the                */
    /* consumption-day amount.  The model allows for covariates and                */
    /* includes a random effect in both parts and allows for correlation           */
    /* between the random effects (Tooze et al., 2006, Journal of the              */
    /* American Dietetic Association, 106, 1575-1587).                             */
    /*                                                                             */
    /* To fit this nonlinear mixed model with correlated random effects            */
    /* (i.e. the correlated model), starting values for the two parts of           */
    /* the model are obtained by first using the GENMOD procedure to fit           */
    /* a probability model and an amount model.  Then a nonlinear mixed            */
    /* model with uncorrelated random effects (i.e. the uncorrelated               */
    /* model) is fit using two calls to the NLMIXED procedure, and the             */
    /* parameter estimates from this model are used as starting values             */
    /* for the correlated model.                                                   */
    /*                                                                             */
    /* For foods consumed every day and nutrients, the MIXTRAN macro               */
    /* uses the GENMOD procedure to calculate starting values and uses             */
    /* the NLMIXED procedure to fit an amount-only model.                          */
    /*                                                                             */
    /* The syntax for calling the macro is:                                        */
    /*                                                                             */
    /* %mixtran(data=, response=, foodtype=, subject=, repeat=,                    */
    /*          covars_prob=, covars_amt=, outlib=, modeltype=,                    */
    /*          lambda=, replicate_var=, seq=,                                     */
    /*          weekend=, vargroup=, numvargroups=,                                */
    /*          start_val1=, start_val2=, start_val3=, vcontrol=,                  */
    /*          nloptions=, titles=, printlevel=)                                  */
    /*                                                                             */
    /*  where                                                                      */
    /*                                                                             */
    /*   "data"         * Specifies the data set to be used.                       */
    /*                                                                             */
    /*   "response"     * Specifies the 24-hour recall variable.                   */
    /*                                                                             */
    /*   "foodtype"     * Specifies a name for the analysis, used to               */
    /*                    identify the output data sets.  This value can           */
    /*                    be the same as the response variable.                    */
    /*                                                                             */
    /*   "subject"      * Specifies the variable that uniquely                     */
    /*                    identifies each subject.                                 */
    /*                                                                             */
    /*   "repeat"       * Specifies the variable that indexes repeated             */
    /*                    observations for each subject.                           */
    /*                                                                             */
    /*   "covars_prob"    Specifies a list of covariates for the first             */
    /*                    part of the model that models the probability            */
    /*                    of consumption.  Covariates must be separated            */
    /*                    by spaces.  Interactions must be in the order            */
    /*                    specified by PROC GENMOD.  If the model type             */
    /*                    is "amount" then covars_prob should be left as           */
    /*                    a null string.                                           */
    /*                                                                             */
    /*   "covars_amt"   * Specifies a list of covariates for the second            */
    /*                    part of the model that models the consumption-           */
    /*                    day amount.  Covariates must be separated by             */
    /*                    spaces.  Interactions must be in the order               */
    /*                    specified by PROC GENMOD.                                */
    /*                                                                             */
    /*   "outlib"       * Specifies a directory where output data sets             */
    /*                    are stored.  Outlib can not be null.                     */
    /*                                                                             */
    /*   "modeltype"    * Specifies the model.  The possible values are:           */
    /*                    "null string" = fit correlated model,                    */
    /*                    "corr"        = fit correlated model,                    */
    /*                    "nocorr"      = fit uncorrelated model,                  */
    /*                    "amount"      = fit amount-only model.                   */
    /*                                                                             */
    /*   "lambda"         Specifies a user supplied value for the                  */
    /*                    Box-Cox transformation parameter, lambda.  If            */
    /*                    a value is not supplied, the macro will                  */
    /*                    calculate a value for lambda.                            */
    /*                                                                             */
    /*   "replicate_var"  Specifies the variable to be used in the                 */
    /*                    replicate statement of PROC NLMIXED or the               */
    /*                    freq statement of PROC UNIVARIATE.  The                  */
    /*                    specified variable must be integer valued.               */
    /*                                                                             */
    /*   "seq"            Specifies one or more sequence indicator                 */
    /*                    variables to account for effects due to the              */
    /*                    sequence number of a subject's records.  This            */
    /*                    variable can NOT also appear in covars_prob              */
    /*                    or covars_amt.                                           */
    /*                                                                             */
    /*   "weekend"        Specifies the weekend (Fri.-Sun.) indicator              */
    /*                    variable to account for a weekend effect.  A             */
    /*                    value of 1 represents a Fri.-Sun. record, and            */
    /*                    a value of 0 represents a Mon.-Thurs. record.            */
    /*                    This variable can NOT also appear in                     */
    /*                    covars_prob or covars_amt.                               */
    /*                                                                             */
    /*   "vargroup"       Specifies a variable that groups observations            */
    /*                    to allow the model to incorporate a separate             */
    /*                    residual variance parameter for each of these            */
    /*                    groups of observations.  If the output from              */
    /*                    this macro is to be used in the DISTRIB macro,           */
    /*                    then only the weekend variable can be used.              */
    /*                                                                             */
    /*   "numvargroups"   Specifies the number of groups defined by the            */
    /*                    vargroup variable.  If the output from this              */
    /*                    macro is to be used in the DISTRIB macro and             */
    /*                    weekend is the "vargroup" variable, then the             */
    /*                    number of groups is 2.                                   */
    /*                                                                             */
    /*   "start_val1"     Starting values for probability model (nocorr).          */
    /*                    Use only when vcontrol is called and parameter           */
    /*                    estimates (i.e. _parmsf1_"foodtype") from a              */
    /*                    previous execution of an analogous model are desired.    */
    /*                    Specifies the starting values data set for the           */
    /*                    1st PROC NLMIXED (i.e. NLMIXED for probability           */
    /*                    model).                                                  */
    /*                                                                             */
    /*   "start_val2"     Starting values for the amount model.                    */
    /*                    Use only when vcontrol is called and parameter           */
    /*                    estimates (i.e. _parmsf2_"foodtype") from a              */
    /*                    previous execution of an analogous model are desired.    */
    /*                    Specifies the starting values data set for the           */
    /*                    2nd PROC NLMIXED (i.e. NLMIXED for amount                */
    /*                    model).                                                  */
    /*                                                                             */
    /*   "start_val3"     Starting values for correlated model (corr).             */
    /*                    Use only when vcontrol and parameter                     */
    /*                    estimates (i.e. _parmsf3_"foodtype") from a              */
    /*                    previous execution of an analogous model are desired.    */
    /*                    Specifies the starting values data set for the           */
    /*                    3rd PROC NLMIXED (i.e. NLMIXED for correlated            */
    /*                    model).                                                  */
    /*                                                                             */
    /*   "vcontrol"       Use only when starting values from a previous            */
    /*                    execution of the same model are also used.               */
    /*                    Specifies a 1 to 6 character name to                     */
    /*                    differentiate output data sets for runs using            */
    /*                    the same food.  See the parameters start_val1,           */
    /*                    start_val2, and start_val3.  The default is              */
    /*                    null.                                                    */
    /*                                                                             */
    /*   "nloptions"      Specifies a list of options to be added to all           */
    /*                    calls to PROC NLMIXED, for example:                      */
    /*                       nloptions=qpoints=1 gconv=1e-12 itdetails.            */
    /*                                                                             */
    /*   "titles"         Specifies the number of title lines (0-4) to             */
    /*                    be reserved for the user's titles.  Up to 4              */
    /*                    title lines may be reserved for the user's               */
    /*                    titles.  The remaining title lines are used by           */
    /*                    the macro.  The default value is 0.                      */
    /*                                                                             */
    /*   "printlevel"     Specifies 1, 2, or 3 to control the amount of            */
    /*                    information printed in the list file.                    */
    /*                    Printlevel=1 prints only the summary reports.            */
    /*                    Printlevel=2 prints summary reports and output           */
    /*                    from the NLMIXED procedures.  Printlevel=2 is            */
    /*                    the default value.  Printlevel=3 prints                  */
    /*                    summary reports and output from all of the               */
    /*                    statistical procedures.                                  */
    /*                                                                             */
    /*                                                                             */
    /* Note:  * Parameters marked with an asterisk are mandatory, so a             */
    /*          value must be supplied in the macro call.                          */
    /*                                                                             */
    /* Caution:  variable name "YN" is reserved for this macro.                    */
    /*                                                                             */
    /* Caution:  data set names "data" and "data0" and "misc_info" are             */
    /*           reserved for this macro.                                          */
    /*                                                                             */
    /*******************************************************************************/

    ************************************************************;
    *** Call MIXTRAN to fit a nonlinear mixed model for fish ***;
    ************************************************************;

    %mixtran(data          = datamrec,
             response      = R,
             foodtype      = fish&replicnum,
             subject       = subjectid,
             repeat        = repeat,
             covars_prob   = race1 race3 dmdeduc1 dmdeduc3 ridageyr ffqeqzero bcffq0mean,
             covars_amt    = race1 race3 dmdeduc1 dmdeduc3 ridageyr ffqeqzero bcffq0mean,
             outlib        = outlib,
             modeltype     = corr,
             lambda        = ,
             replicate_var = &replicvar,
             seq           = ,
             weekend       = ,
             vargroup      = ,
             numvargroups  = ,
             subgroup      = ,
             start_val1    = ,
             start_val2    = ,
             start_val3    = ,
             vcontrol      = ,
             nloptions     = qmax=61,
             titles        = 4,
             printlevel    = 1
             );



    ********************************************************;
    *** If the MIXTRAN macro executed successfully then  ***;
    **  continue with this iteration of the example,     ***;
    **  otherwise stop this iteration and begin the next ***;
    ********************************************************;

    %if &success = 1 %then %do;



      **********************************************************;
      *** Call the DISTRIB and BOXCOX_SURVEY macros to find  ***;
      *** the optimal value of the Box-Cox transformation    ***;
      *** parameter. (For details, see documentation for the ***;
      *** DISTRIB and BOXCOX_SURVEY macros)                  ***;
      **********************************************************;

      %let loopseed_distrib = %eval(89009890 + &replicnum * 10000);

      %distrib(call_type  = ,
               seed       = &loopseed_distrib,
               nsim_mc    = 100,
               modeltype  = corr,
               pred       = outlib._pred_fish&replicnum,
               param      = outlib._param_fish&replicnum,
               outlib     = work,
               cutpoints  = ,
               ncutpnt    = ,
               byvar      = ,
               subgroup   = ,
               add_da     = ,
               subject    = subjectid,
               titles     = 4,
               food       = fish,
               mcsimda    = mcsim,
               recamt     = ,
               recamt_co  = ,
               recamt_hi  = ,
               wkend_prop = ,
               wkend_mc   =
               );

      %boxcox_survey(data    = mcsim,
                     subject = subjectid,
                     var     = mc_t,
                     weight  = mcsim_wt,
                     print   = Y,
                     plot    = N,
                     ntitle  = 4
                     );

      data _null_;
        set mcsim(obs=1);
        call symput('boxcox_t_param', lambda_mc_t);
      run;



      *******************************************************;
      *** Create an input data set for the INDIVINT macro ***;
      *******************************************************;

      data parampred(keep = subjectid p_var_u1 a_var_u2 cov_u1u2 a_var_e a_lambda x1b1 x2b2);
        if (_n_ = 1) then set outlib._param_fish&replicnum;
        set outlib._pred_fish&replicnum;
      run;

      data parsubj1rec;
        merge parampred subj1rec(keep = subjectid R1 R2);
        by subjectid;
      run;

      data paramsubj1rec;
        if _n_ = 1 then set outmin_amt;
        set parsubj1rec;

        lamt = &boxcox_t_param;
      run;



      /*************************************************************************/
      /*************************************************************************/
      /*                                                                       */
      /*  Description of the INDIVINT Macro                                    */
      /*                                                                       */
      /*************************************************************************/
      /*                                                                       */
      /*                                                                       */
      /* The INDIVINT macro calculates predicted values for regression         */
      /* calibration using methods from Kipnis et al. (Biometrics, 2009) and   */
      /* using results from an amount-only model or a two-part model fit using */
      /* the MIXTRAN macro.  The INDIVINT macro performs adaptive Gaussian     */
      /* quadrature to predict usual intake for each individual, and the macro */
      /* allows the user to provide a Box-Cox transformation parameter in      */
      /* order to calculate the predicted values on a transformed scale.  The  */
      /* results from this macro are intended for use in a subsequent          */
      /* regression model as discussed by Kipnis et al. (Biometrics, 2009).    */
      /*                                                                       */
      /* The syntax for calling the INDIVINT macro is:                         */
      /*                                                                       */
      /* %indivint(model12=, subj1recdata=, recid=, r24vars=, min_amt=,        */
      /*           var_u1=, var_u2=, cov_u1u2=, var_e=, lambda=, xbeta1=,      */
      /*           xbeta2=, boxcox_t_lamt=, lamt=, dencalc=, denopt=,          */
      /*           u1nlmix=, u2nlmix=, titles=, notesprt=);                    */
      /*                                                                       */
      /* where the parameters are described as follows.                        */
      /*                                                                       */
      /*  "model12"            Specifies the type of model that was fit prior  */
      /*                       to calling this macro.  A value of 1 indicates  */
      /*                       that an amount-only model was fit, and a value  */
      /*                       of 2 indicates that a two-part model was fit    */
      /*                       where part 1 is the probability part of the     */
      /*                       model and part 2 is the amount part of the      */
      /*                       model.                                          */
      /*                                                                       */
      /*  "subj1recdata"       Specifies a data set with 1 record for each     */
      /*                       individual.                                     */
      /*                                                                       */
      /*  "recid"              Specifies an identification (ID) variable that  */
      /*                       uniquely identifies each individual's record.   */
      /*                                                                       */
      /*  "r24vars"            Specifies the 24-hour recall variables with     */
      /*                       values that are either non-negative or a SAS    */
      /*                       missing value if the 24-hour recall is not      */
      /*                       available.  Variables must be space delimited   */
      /*                       as illustrated in the following example:        */
      /*                       "r24vars=r24hr1 r24hr2".                        */
      /*                       Note for Advanced Users:  If all 24-hour recall */
      /*                       values are missing for each subject, then the   */
      /*                       denominator integration should not be           */
      /*                       performed, so the "dencalc" macro parameter     */
      /*                       should be specified as "dencalc=n".             */
      /*                                                                       */
      /*  "min_amt"            Specifies a variable that provides the minimum  */
      /*                       intake amount.  This value may be selected as   */
      /*                       the smallest value among the observed           */
      /*                       consumption-day amounts.  Note that the         */
      /*                       specified variable provides the same value for  */
      /*                       each individual.  This value will be divided in */
      /*                       half and used in the calculations for the       */
      /*                       numerator integration.                          */
      /*                                                                       */
      /*  "var_u1"             Specifies a variable that provides the variance */
      /*                       estimate for u1, the random effect from the     */
      /*                       probability part of the model.  If a variable   */
      /*                       is specified, then the macro will use its value */
      /*                       as a diagonal entry of the covariance matrix    */
      /*                       which is either a 1x1 matrix or a 2x2 matrix    */
      /*                       depending on the number of random effects that  */
      /*                       are in the model.                               */
      /*                                                                       */
      /*  "var_u2"             Specifies a variable that provides the variance */
      /*                       estimate for u2, the random effect from the     */
      /*                       amount part of the model or from an amount-only */
      /*                       model.  If a variable is specified, then the    */
      /*                       macro will use its value as a diagonal entry of */
      /*                       the covariance matrix which is either a 1x1     */
      /*                       matrix or a 2x2 matrix depending on the number  */
      /*                       of random effects that are in the model.        */
      /*                                                                       */
      /*  "cov_u1u2"           Specifies a variable that provides the estimate */
      /*                       of the covariance(u1, u2) from the two-part     */
      /*                       model.  If the two-part model was an            */
      /*                       uncorrelated model, then the specified variable */
      /*                       should have a value of zero for every           */
      /*                       individual's record.                            */
      /*                                                                       */
      /*  "var_e"              Specifies a variable that provides the variance */
      /*                       estimate for e, the within-person error term    */
      /*                       from the amount part of the model or from an    */
      /*                       amount-only model.                              */
      /*                                                                       */
      /*  "lambda"             Specifies a variable that provides the estimate */
      /*                       of the Box-Cox parameter, lambda, from the      */
      /*                       amount part of the model or from an amount-only */
      /*                       model.                                          */
      /*                                                                       */
      /*  "xbeta1"             Specifies a variable that provides the linear   */
      /*                       predictor values calculated using the           */
      /*                       covariates and estimates of the fixed effects   */
      /*                       parameters from the probability part of the     */
      /*                       model.                                          */
      /*                                                                       */
      /*  "xbeta2"             Specifies a variable that provides the linear   */
      /*                       predictor values calculated using the           */
      /*                       covariates and estimates of the fixed effects   */
      /*                       parameters from the amount part of the model or */
      /*                       from an amount-only model.                      */
      /*                                                                       */
      /*  "boxcox_t_lamt"      If "boxcox_t_lamt=y" or "boxcox_t_lamt=Y" then  */
      /*                       individual usual intake will be predicted on a  */
      /*                       transformed scale where the Box-Cox             */
      /*                       transformation is used with the Box-Cox         */
      /*                       parameter value provided by the "lamt" macro    */
      /*                       parameter.  The default value for               */
      /*                       "boxcox_t_lamt" is "n".                         */
      /*                                                                       */
      /*  "lamt"               Specifies a variable that provides the Box-Cox  */
      /*                       parameter value when "boxcox_t_lamt=y" or       */
      /*                       "boxcox_t_lamt=Y".  The macro does not allow    */
      /*                       the Box-Cox parameter to be negative.           */
      /*                                                                       */
      /*  "dencalc"            By default, "dencalc=y" so the denominator      */
      /*                       integration is performed.                       */
      /*                       Note for Advanced Users:  If all 24-hour recall */
      /*                       variables are missing for each subject, then    */
      /*                       the denominator integration should not be       */
      /*                       performed, so the "dencalc" option should be    */
      /*                       specified as "dencalc=n".                       */
      /*                                                                       */
      /*  "denopt"             By default, "denopt=y" so the denominator       */
      /*                       optimization is performed as part of the        */
      /*                       denominator integration calculations.           */
      /*                       Note for Advanced Users:  In some situations    */
      /*                       the denominator optimization is redundant       */
      /*                       because the empirical Bayes estimates of u1 and */
      /*                       u2 are available from the model fitting         */
      /*                       software; therefore, in these situations,       */
      /*                       setting "denopt=n" or "denopt=N" allows the     */
      /*                       macro to skip this optimization step and use    */
      /*                       the variables provided by the "u1nlmix" and     */
      /*                       "u2nlmix" macro parameters.                     */
      /*                                                                       */
      /*  "u1nlmix"            Specifies a variable for an Advanced Users      */
      /*                       option.  For details, see the description for   */
      /*                       the "denopt" macro parameter.                   */
      /*                                                                       */
      /*  "u2nlmix"            Specifies a variable for an Advanced Users      */
      /*                       option.  For details, see the description for   */
      /*                       the "denopt" macro parameter.                   */
      /*                                                                       */
      /*  "titles"             Specifies the number of title lines to be       */
      /*                       reserved for the user's titles.  One additional */
      /*                       title line is used by the macro.  The default   */
      /*                       value is "0".                                   */
      /*                                                                       */
      /*  "notesprt"           If "notesprt=n" or "notesprt=N" then notes are  */
      /*                       not printed to the SAS log.  The default value  */
      /*                       for "notesprt" is "y".                          */
      /*                                                                       */
      /*************************************************************************/

      *******************************************************;
      *** Call INDIVINT to predict the intake of red meat ***;
      *******************************************************;

      %indivint(model12       = 2,
                subj1recdata  = paramsubj1rec,
                recid         = subjectid,
                r24vars       = R1 R2,
                min_amt       = min_amt,
                var_u1        = p_var_u1,
                var_u2        = a_var_u2,
                cov_u1u2      = cov_u1u2,
                var_e         = a_var_e,
                lambda        = a_lambda,
                xbeta1        = x1b1,
                xbeta2        = x2b2,
                boxcox_t_lamt = y,
                lamt          = lamt,
                dencalc       = y,
                denopt        = y,
                u1nlmix       = ,
                u2nlmix       = ,
                titles        = 4,
                notesprt      = y
                );



      ****************************************************************;
      *** Create a data set with the INDIVINT results for red meat ***;
      ****************************************************************;

      data subj1recres;
        merge subj1rec _resdata(keep = subjectid indusint);
          by subjectid;
      run;



      *****************************************************************************************************;
      *** Standardize the covariates so that one unit equals the distance between transformed t0 and t1 ***;
      *****************************************************************************************************;

      data subj1recres;
        set subj1recres;

        lamt = &boxcox_t_param;

        bc_t0 = (&t0**lamt - 1) / lamt;
        bc_t1 = (&t1**lamt - 1) / lamt;

        unit = bc_t1 - bc_t0;

        indusint_unit = indusint / unit;
      run;



      *************************************;
      *** Fit a linear regression model ***;
      *************************************;

      proc reg data=subj1recres outest=regout;
        freq &replicvar;
        model logtotmerc = indusint_unit;
        title5 "Linear Regression Model: Adjusted for Measurement Error";
      run;
      title5;



      *************************************************************;
      *** Save the regression results for the current replicate ***;
      *************************************************************;

      data outlib.diethealthout&replicnum (keep = replicate intercept rc_beta_unit lamt t0 t1);

        set regout(rename = (indusint_unit=rc_beta_unit));

        replicate = &replicnum;

        ***********************************************;
        *** Save the comparison values and lambda t ***;
        ***********************************************;

        lamt = &boxcox_t_param;

        t0 = &t0;
        t1 = &t1;

      run;

    ******************************************************************;
    *** End of the processing of the INDIVINT macro for iterations ***;
    *** that were successful in the MIXTRAN macro                  ***;
    ******************************************************************;

    %end;



    %else %do;

      ************************************************;
      *** Exit the loop if MIXTRAN is unsuccessful ***;
      ************************************************;

      %put MIXTRAN was not successful for replicate data set &replicnum - INDIVINT macro was not executed;

    %end;

  %end;   *** End of the replicate loop ***;

%mend fit_models_replicate_loop;



************************************************;
*** Call the fit_models_replicate_loop macro ***;
************************************************;

%fit_models_replicate_loop;
